/*
 * Decompiled with CFR 0.152.
 */
package com.schematic.api.core;

import java.io.IOException;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Optional;
import java.util.Random;
import okhttp3.Interceptor;
import okhttp3.Response;

public class RetryInterceptor
implements Interceptor {
    private static final Duration INITIAL_RETRY_DELAY = Duration.ofMillis(1000L);
    private static final Duration MAX_RETRY_DELAY = Duration.ofMillis(60000L);
    private static final double JITTER_FACTOR = 0.2;
    private final ExponentialBackoff backoff;
    private final Random random = new Random();

    public RetryInterceptor(int maxRetries) {
        this.backoff = new ExponentialBackoff(maxRetries);
    }

    public Response intercept(Interceptor.Chain chain) throws IOException {
        Response response = chain.proceed(chain.request());
        if (RetryInterceptor.shouldRetry(response.code())) {
            return this.retryChain(response, chain);
        }
        return response;
    }

    private Response retryChain(Response response, Interceptor.Chain chain) throws IOException {
        Optional<Duration> nextBackoff = this.backoff.nextBackoff(response);
        while (nextBackoff.isPresent()) {
            try {
                Thread.sleep(nextBackoff.get().toMillis());
            }
            catch (InterruptedException e) {
                throw new IOException("Interrupted while trying request", e);
            }
            response.close();
            response = chain.proceed(chain.request());
            if (RetryInterceptor.shouldRetry(response.code())) {
                nextBackoff = this.backoff.nextBackoff(response);
                continue;
            }
            return response;
        }
        return response;
    }

    private Duration getRetryDelayFromHeaders(Response response, int retryAttempt) {
        Optional<Duration> rateLimitDelay;
        String rateLimitReset;
        String retryAfter = response.header("Retry-After");
        if (retryAfter != null) {
            Optional<Duration> secondsDelay = this.tryParseLong(retryAfter).map(seconds -> seconds * 1000L).filter(delayMs -> delayMs > 0L).map(delayMs -> Math.min(delayMs, MAX_RETRY_DELAY.toMillis())).map(Duration::ofMillis);
            if (secondsDelay.isPresent()) {
                return secondsDelay.get();
            }
            Optional<Duration> dateDelay = this.tryParseHttpDate(retryAfter).map(resetTime -> resetTime.toInstant().toEpochMilli() - System.currentTimeMillis()).filter(delayMs -> delayMs > 0L).map(delayMs -> Math.min(delayMs, MAX_RETRY_DELAY.toMillis())).map(Duration::ofMillis);
            if (dateDelay.isPresent()) {
                return dateDelay.get();
            }
        }
        if ((rateLimitReset = response.header("X-RateLimit-Reset")) != null && (rateLimitDelay = this.tryParseLong(rateLimitReset).map(resetTimeSeconds -> resetTimeSeconds * 1000L - System.currentTimeMillis()).filter(delayMs -> delayMs > 0L).map(delayMs -> Math.min(delayMs, MAX_RETRY_DELAY.toMillis())).map(this::addPositiveJitter).map(Duration::ofMillis)).isPresent()) {
            return rateLimitDelay.get();
        }
        long baseDelay = INITIAL_RETRY_DELAY.toMillis() * (1L << retryAttempt);
        long cappedDelay = Math.min(baseDelay, MAX_RETRY_DELAY.toMillis());
        return Duration.ofMillis(this.addSymmetricJitter(cappedDelay));
    }

    private Optional<Long> tryParseLong(String value) {
        if (value == null) {
            return Optional.empty();
        }
        try {
            return Optional.of(Long.parseLong(value));
        }
        catch (NumberFormatException e) {
            return Optional.empty();
        }
    }

    private Optional<ZonedDateTime> tryParseHttpDate(String value) {
        if (value == null) {
            return Optional.empty();
        }
        try {
            return Optional.of(ZonedDateTime.parse(value, DateTimeFormatter.RFC_1123_DATE_TIME));
        }
        catch (DateTimeParseException e) {
            return Optional.empty();
        }
    }

    private long addPositiveJitter(long delayMs) {
        double jitterMultiplier = 1.0 + this.random.nextDouble() * 0.2;
        return (long)((double)delayMs * jitterMultiplier);
    }

    private long addSymmetricJitter(long delayMs) {
        double jitterMultiplier = 1.0 + (this.random.nextDouble() - 0.5) * 0.2;
        return (long)((double)delayMs * jitterMultiplier);
    }

    private static boolean shouldRetry(int statusCode) {
        return statusCode == 408 || statusCode == 429 || statusCode >= 500;
    }

    private final class ExponentialBackoff {
        private final int maxNumRetries;
        private int retryNumber = 0;

        ExponentialBackoff(int maxNumRetries) {
            this.maxNumRetries = maxNumRetries;
        }

        public Optional<Duration> nextBackoff(Response response) {
            if (this.retryNumber >= this.maxNumRetries) {
                return Optional.empty();
            }
            Duration delay = RetryInterceptor.this.getRetryDelayFromHeaders(response, this.retryNumber);
            ++this.retryNumber;
            return Optional.of(delay);
        }
    }
}

