package com.selectdb.load;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.selectdb.exceptions.CopyIntoException;
import com.selectdb.model.BaseResponse;
import com.selectdb.model.CopyIntoResp;
import com.selectdb.model.CopyIntoResult;
import com.selectdb.utils.Preconditions;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.StringJoiner;
import java.util.regex.Pattern;

import static com.selectdb.model.CopyIntoResult.FINISHED_STATE;

public class CopyExecutor implements Serializable {
    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LoggerFactory.getLogger(CopyExecutor.class);
    private static final String COPY_URL_PATTERN = "http://%s:%s/copy/query";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    public static final Pattern COMMITTED_PATTERN =
            Pattern.compile("errCode = 2, detailMessage = No files can be copied.*");
    public static final int EXEC_SUCCESS = 0;
    private String copyUrl;
    private String clusterName;
    private String username;
    private String password;
    private String tableIdentifier;
    private Map<String, String> properties;
    private HttpClientBuilder httpClientBuilder = new HttpUtil().getHttpClientBuilder();

    public CopyExecutor(LoadOptions options){
        if(options.getDatabase() != null && options.getTable() != null){
            this.tableIdentifier = options.getDatabase() + "." + options.getTable();
        }
        this.properties = options.getPropertyMap();
        this.copyUrl = String.format(COPY_URL_PATTERN, options.getHost(), options.getHttpPort());
        this.username = options.getUsername();
        this.password = options.getPassword();
        this.clusterName = options.getClusterName();
    }

    public BaseResponse execute(String fileName) throws CopyIntoException {
        String sql = buildCopySQL(fileName);
        BaseResponse copyResp = executeCopy(sql);
        return copyResp;
    }

    public String buildCopySQL(String file){
        Preconditions.checkNotNullOrEmpty(tableIdentifier, "Please specify the table name to write.");
        StringBuilder sb = new StringBuilder();
        sb.append("COPY INTO ")
                .append(tableIdentifier)
                .append(" FROM @~('{").append(file).append("}') ")
                .append("PROPERTIES (");

        StringJoiner props = new StringJoiner(",");
        for(Map.Entry<String,String> entry : properties.entrySet()){
            String prop = String.format("'%s'='%s'", entry.getKey(), entry.getValue());
            props.add(prop);
        }
        sb.append(props).append(")");
        return sb.toString();
    }

    public BaseResponse executeCopy(String sql) throws CopyIntoException {
        BaseResponse copyResp = null;
        LOG.info("commit to cluster {} with copy into sql: {}", clusterName, sql);

        int statusCode = -1;
        String reasonPhrase = null;
        Map<String,String> params = new HashMap<>();
        params.put("cluster", clusterName);
        params.put("sql", sql);
        String loadResult = "";
        HttpPostBuilder postBuilder = new HttpPostBuilder();
        try {
            postBuilder.setUrl(copyUrl)
                    .baseAuth(username, password)
                    .setEntity(new StringEntity(OBJECT_MAPPER.writeValueAsString(params)));
            try(CloseableHttpResponse response = httpClientBuilder.build().execute(postBuilder.build())) {
                statusCode = response.getStatusLine().getStatusCode();
                reasonPhrase = response.getStatusLine().getReasonPhrase();
                if (statusCode != 200) {
                    LOG.error("copy into failed with status {}, reason {}", statusCode, reasonPhrase);
                    copyResp = BaseResponse.fail(statusCode, "execute copy into api failed, reason " + reasonPhrase);
                } else if (response.getEntity() != null){
                    loadResult = EntityUtils.toString(response.getEntity());
                    copyResp = handleCommitResponse(loadResult);
                }
            }
        } catch (IOException e) {
            LOG.error("Execute copy into error,", e);
            throw new CopyIntoException("Execute copy into error");
        }
        return copyResp;
    }

    private BaseResponse handleCommitResponse(String loadResult) throws CopyIntoException {
        BaseResponse baseResponse = null;
        try {
            baseResponse = OBJECT_MAPPER.readValue(loadResult, BaseResponse.class);
        } catch (JsonProcessingException e) {
            LOG.error("parse copy into response failed, result: {}", loadResult, e);
            throw new CopyIntoException("Failed to execute copy into");
        }
        if(baseResponse.getCode() == EXEC_SUCCESS){
            if(baseResponse.getData() instanceof Map){
                CopyIntoResp dataResp = OBJECT_MAPPER.convertValue(baseResponse.getData(), CopyIntoResp.class);
                CopyIntoResult result = dataResp.getResult();
                if(!FINISHED_STATE.equals(result.getState()) && !isCommitted(result.getMsg())){
                    LOG.error("copy into load failed, reason:{}", loadResult);
                }
                baseResponse.setData(dataResp);
            }
        }else{
            LOG.error("commit failed, reason:{}", loadResult);
        }
        return baseResponse;
    }

    private boolean isCommitted(String msg){
        return COMMITTED_PATTERN.matcher(msg).matches();
    }

    public void setHttpClientBuilder(HttpClientBuilder httpClientBuilder) {
        this.httpClientBuilder = httpClientBuilder;
    }

    public void setTableIdentifier(String tableIdentifier) {
        this.tableIdentifier = tableIdentifier;
    }
}
