package com.selectdb.load;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.selectdb.exceptions.UploadException;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.Serializable;

public class StageLoad implements Serializable {
    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LoggerFactory.getLogger(StageLoad.class);
    private static final String UPLOAD_URL_PATTERN = "http://%s:%s/copy/upload";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    static {
        OBJECT_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }
    private String uploadUrl;
    private String username;
    private String password;
    private HttpClientBuilder httpClientBuilder = new HttpUtil().getHttpClientBuilder();

    public StageLoad(LoadOptions options){
        this.uploadUrl = String.format(UPLOAD_URL_PATTERN, options.getHost(), options.getHttpPort());
        this.username = options.getUsername();
        this.password = options.getPassword();
    }

    /**
     * upload to object storage
     */
    public void uploadToStorage(String fileName, byte[] data) throws UploadException{
        long start = System.currentTimeMillis();
        LOG.info("file write started for {}", fileName);
        String address = getUploadAddress(fileName);
        long addressTs = System.currentTimeMillis();
        LOG.info("redirect to internalStage address:{}, cost {} ms", address, addressTs - start);
        String requestId = uploadToInternalStage(address, data);
        LOG.info("upload file {} finished, size {}, cost {}ms, with requestId {}", fileName,  data.length, System.currentTimeMillis() - addressTs, requestId);
    }

    public String uploadToInternalStage(String address, byte[] data) throws UploadException {
        ByteArrayEntity entity = new ByteArrayEntity(data);
        HttpPutBuilder putBuilder = new HttpPutBuilder();
        putBuilder.setUrl(address)
                .addCommonHeader()
                .setEntity(entity);
        HttpPut httpPut = putBuilder.build();
        try{
            try(CloseableHttpResponse response = httpClientBuilder.build().execute(httpPut)){
                final int statusCode = response.getStatusLine().getStatusCode();
                String requestId = getRequestId(response.getAllHeaders());
                if (statusCode == 200 && response.getEntity() != null) {
                    String loadResult = EntityUtils.toString(response.getEntity());
                    if(loadResult == null || loadResult.isEmpty()){
                        //upload finished
                        return requestId;
                    }
                    LOG.error("upload file failed, requestId is {}, response result: {}", requestId, loadResult);
                    throw new UploadException("upload file failed: " + response.getStatusLine().toString() + ", with requestId " + requestId);
                }
                throw new UploadException("upload file error: " + response.getStatusLine().toString() + ", with requestId " + requestId);
            }
        } catch (IOException ex){
            LOG.error("Failed to upload data to internal stage ", ex);
            throw new UploadException("Failed to upload data to internal stage");
        }
    }

    /**
     * get requestId from response Header for upload stage
     * header key are: x-oss-request-id/x-cos-request-id/x-obs-request-id/x-amz-request-id
     * @return key:value
     */
    public String getRequestId(Header[] headers){
        if(headers == null || headers.length == 0){
            return null;
        }
        for (int i = 0; i < headers.length; i++) {
            final Header header = headers[i];
            String name = header.getName();
            if (name != null && name.toLowerCase().matches("x-\\S+-request-id")) {
                return name + ":" + header.getValue();
            }
        }
        return null;
    }

    /**
     * Get the redirected s3 address
     */
    public String getUploadAddress(String fileName) throws UploadException{
        HttpPutBuilder putBuilder = new HttpPutBuilder();
        putBuilder.setUrl(uploadUrl)
                .addFileName(fileName)
                .addCommonHeader()
                .setEmptyEntity()
                .baseAuth(username, password);
        try{
            try(CloseableHttpResponse execute = httpClientBuilder.build().execute(putBuilder.build())){
                int statusCode = execute.getStatusLine().getStatusCode();
                String reason = execute.getStatusLine().getReasonPhrase();
                if(statusCode == 307){
                    Header location = execute.getFirstHeader("location");
                    String uploadAddress = location.getValue();
                    return uploadAddress;
                }else {
                    HttpEntity entity = execute.getEntity();
                    String result = entity == null ? null : EntityUtils.toString(entity);
                    LOG.error("Failed to get internalStage address, status {}, reason {}, response {}", statusCode, reason, result);
                    throw new UploadException("Failed get internalStage address");
                }
            }
        }catch (IOException e){
            LOG.error("get internalStage address error,", e);
            throw new UploadException("get internalStage address error");
        }
    }

    public void setHttpClientBuilder(HttpClientBuilder httpClientBuilder) {
        this.httpClientBuilder = httpClientBuilder;
    }
}
