/*
 * Decompiled with CFR 0.152.
 */
package com.stimulussoft.filequeue.processor;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.util.concurrent.MoreExecutors;
import com.stimulussoft.filequeue.FileQueueItem;
import com.stimulussoft.filequeue.processor.Consumer;
import com.stimulussoft.filequeue.processor.DelayRejectPolicy;
import com.stimulussoft.filequeue.processor.Expiration;
import com.stimulussoft.filequeue.store.MVStoreQueue;
import com.stimulussoft.util.ThreadUtil;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Phaser;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueueProcessor<T> {
    private static final Logger logger = LoggerFactory.getLogger(QueueProcessor.class);
    private static final ThreadPoolExecutor executorService = new ThreadPoolExecutor(Runtime.getRuntime().availableProcessors(), Runtime.getRuntime().availableProcessors() * 8, 60L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>(true), ThreadUtil.getFlexibleThreadFactory("filequeue-worker", false), new DelayRejectPolicy());
    private static final ScheduledExecutorService mvstoreCleanUPScheduler = Executors.newScheduledThreadPool(Runtime.getRuntime().availableProcessors(), ThreadUtil.getFlexibleThreadFactory("mvstore-cleanup", true));
    private final ObjectMapper objectMapper;
    private final MVStoreQueue mvStoreQueue;
    private final Class<T> type;
    private final Consumer<T> consumer;
    private final Expiration<T> expiration;
    private final Phaser restorePolled = new Phaser();
    private Optional<ScheduledFuture<?>> cleanupTaskScheduler;
    private volatile boolean doRun = true;
    private final int maxTries;
    private final int retryDelay;
    private final int persistRetryDelay;
    private final int maxRetryDelay;
    private final Path queuePath;
    private final String queueName;
    private final TimeUnit retryDelayUnit;
    private final TimeUnit persistRetryDelayUnit;
    private final RetryDelayAlgorithm retryDelayAlgorithm;

    public static void destroy() {
        MoreExecutors.shutdownAndAwaitTermination((ExecutorService)executorService, (long)60L, (TimeUnit)TimeUnit.SECONDS);
        MoreExecutors.shutdownAndAwaitTermination((ExecutorService)mvstoreCleanUPScheduler, (long)60L, (TimeUnit)TimeUnit.SECONDS);
    }

    public static Builder builder(String queueName, Path queuePath, Class type, Consumer consumer) throws IllegalArgumentException {
        return new Builder(queueName, queuePath, type, consumer);
    }

    public static Builder builder() {
        return new Builder();
    }

    QueueProcessor(Builder builder) throws IOException, IllegalStateException, IllegalArgumentException {
        if (builder.queueName == null) {
            throw new IllegalArgumentException("queue name must be specified");
        }
        if (builder.queuePath == null) {
            throw new IllegalArgumentException("queue path must be specified");
        }
        if (builder.type == null) {
            throw new IllegalArgumentException("item type must be specified");
        }
        if (builder.consumer == null) {
            throw new IllegalArgumentException("consumer must be specified");
        }
        this.objectMapper = this.createObjectMapper();
        if (!this.objectMapper.canSerialize(builder.type)) {
            throw new IllegalArgumentException("The given type is not serializable. it cannot be serialized by jackson");
        }
        this.queueName = builder.queueName;
        this.queuePath = builder.queuePath;
        this.consumer = builder.consumer;
        this.expiration = builder.expiration;
        this.type = builder.type;
        this.maxTries = builder.maxTries;
        this.retryDelay = builder.retryDelay;
        this.retryDelayUnit = builder.retryDelayUnit;
        this.maxRetryDelay = builder.maxRetryDelay;
        this.retryDelayAlgorithm = builder.retryDelayAlgorithm;
        this.mvStoreQueue = new MVStoreQueue(builder.queuePath, builder.queueName);
        this.persistRetryDelay = builder.persistRetryDelay <= 0 ? (this.retryDelay <= 1 ? 1 : this.retryDelay / 2) : builder.persistRetryDelay;
        this.persistRetryDelayUnit = builder.persistRetryDelayUnit;
        this.cleanupTaskScheduler = Optional.of(mvstoreCleanUPScheduler.scheduleWithFixedDelay(new MVStoreCleaner(this), 0L, this.persistRetryDelay, this.retryDelayUnit));
    }

    private static long dateDiff(Date date1, Date date2, TimeUnit unit) {
        long diffInMillies = date2.getTime() - date1.getTime();
        return unit.convert(diffInMillies, TimeUnit.MILLISECONDS);
    }

    public Path getQueueBaseDir() {
        return this.mvStoreQueue.getQueueDir();
    }

    public void reopen() throws IllegalStateException {
        this.mvStoreQueue.reopen();
    }

    public void submit(T item) throws IllegalStateException, IOException {
        if (!this.doRun) {
            throw new IllegalStateException("file queue {" + this.getQueueBaseDir() + "} is not running");
        }
        try {
            this.restorePolled.register();
            executorService.execute(new ProcessItem<T>(this.consumer, this.expiration, item, this));
        }
        catch (CancellationException | RejectedExecutionException cancel) {
            this.mvStoreQueue.push(this.objectMapper.writeValueAsBytes(item));
        }
        finally {
            this.restorePolled.arriveAndDeregister();
        }
    }

    public void close() {
        this.doRun = false;
        this.cleanupTaskScheduler.ifPresent(cleanupTask -> cleanupTask.cancel(true));
        this.restorePolled.register();
        this.restorePolled.arriveAndAwaitAdvance();
        this.mvStoreQueue.close();
    }

    public long size() {
        return this.mvStoreQueue.size();
    }

    private void tryItem(T item) {
        ((FileQueueItem)item).setTryDate(new Date());
        ((FileQueueItem)item).incTryCount();
    }

    private ObjectMapper createObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        return mapper;
    }

    private boolean isNeedRetry(T item) {
        if (this.maxTries <= 0) {
            return true;
        }
        FileQueueItem queueItem = (FileQueueItem)item;
        return queueItem.getTryCount() < this.maxTries;
    }

    private boolean isTimeToRetry(T item) {
        switch (this.retryDelayAlgorithm) {
            case EXPONENTIAL: {
                int tryDelay = (int)Math.round(Math.pow(2.0, ((FileQueueItem)item).getTryCount()));
                tryDelay = tryDelay > this.maxRetryDelay ? this.maxRetryDelay : tryDelay;
                tryDelay = tryDelay < this.retryDelay ? this.retryDelay : tryDelay;
                return this.isTimeToRetry(item, tryDelay);
            }
        }
        return this.isTimeToRetry(item, this.retryDelay);
    }

    private boolean isTimeToRetry(T item, int retryDelay) {
        return ((FileQueueItem)item).getTryDate() == null || QueueProcessor.dateDiff(((FileQueueItem)item).getTryDate(), new Date(), this.retryDelayUnit) > (long)retryDelay;
    }

    private T deserialize(byte[] data) {
        if (data == null) {
            return null;
        }
        try {
            return (T)this.objectMapper.readValue(data, this.type);
        }
        catch (IOException e) {
            logger.error("failed deserialize object {" + Arrays.toString(data) + "}", (Throwable)e);
            return null;
        }
    }

    public Path getQueuePath() {
        return this.queuePath;
    }

    public String getQueueName() {
        return this.queueName;
    }

    public Consumer getConsumer() {
        return this.consumer;
    }

    public Class getType() {
        return this.type;
    }

    public int getMaxTries() {
        return this.maxTries;
    }

    public int getRetryDelay() {
        return this.retryDelay;
    }

    public int getMaxRetryDelay() {
        return this.maxRetryDelay;
    }

    public TimeUnit getRetryDelayUnit() {
        return this.retryDelayUnit;
    }

    public RetryDelayAlgorithm getRetryDelayAlgorithm() {
        return this.retryDelayAlgorithm;
    }

    public Expiration getExpiration() {
        return this.expiration;
    }

    public int getPersistRetryDelay() {
        return this.persistRetryDelay;
    }

    public TimeUnit getPersistRetryDelayUnit() {
        return this.retryDelayUnit;
    }

    static {
        MoreExecutors.addDelayedShutdownHook((ExecutorService)executorService, (long)60L, (TimeUnit)TimeUnit.SECONDS);
        MoreExecutors.addDelayedShutdownHook((ExecutorService)mvstoreCleanUPScheduler, (long)60L, (TimeUnit)TimeUnit.SECONDS);
    }

    private final class MVStoreCleaner
    implements Runnable {
        private final QueueProcessor queueProcessor;

        MVStoreCleaner(QueueProcessor queueProcessor2) {
            this.queueProcessor = queueProcessor2;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            if (QueueProcessor.this.doRun && !QueueProcessor.this.mvStoreQueue.isEmpty()) {
                try {
                    byte[] toDeserialize;
                    while ((toDeserialize = QueueProcessor.this.mvStoreQueue.poll()) != null) {
                        QueueProcessor.this.restorePolled.register();
                        try {
                            if (!QueueProcessor.this.doRun) {
                                QueueProcessor.this.mvStoreQueue.push(toDeserialize);
                                break;
                            }
                            Object item = QueueProcessor.this.deserialize(toDeserialize);
                            if (item == null) continue;
                            if (QueueProcessor.this.isNeedRetry(item)) {
                                if (QueueProcessor.this.isTimeToRetry(item)) {
                                    this.queueProcessor.submit(item);
                                    continue;
                                }
                                QueueProcessor.this.mvStoreQueue.push(toDeserialize);
                                continue;
                            }
                            if (QueueProcessor.this.expiration == null) continue;
                            QueueProcessor.this.expiration.expire(item);
                        }
                        catch (IllegalStateException e) {
                            logger.error("Failed to process item.", (Throwable)e);
                            QueueProcessor.this.mvStoreQueue.push(toDeserialize);
                        }
                        finally {
                            QueueProcessor.this.restorePolled.arriveAndDeregister();
                        }
                    }
                }
                catch (Exception io) {
                    logger.error("Failed to process item.", (Throwable)io);
                }
                finally {
                    QueueProcessor.this.mvStoreQueue.commit();
                }
            }
        }
    }

    private class ProcessItem<T>
    implements Runnable {
        private final Consumer<T> consumer;
        private final Expiration<T> expiration;
        private final T item;
        private final QueueProcessor<T> queueProcessor;
        private boolean pushback = false;

        ProcessItem(Consumer<T> consumer, Expiration<T> expiration, T item, QueueProcessor<T> queueProcessor2) {
            this.consumer = consumer;
            this.expiration = expiration;
            this.item = item;
            this.queueProcessor = queueProcessor2;
        }

        private void pushBackIfNeeded() {
            if (this.isPushBack()) {
                try {
                    QueueProcessor.this.mvStoreQueue.push(QueueProcessor.this.objectMapper.writeValueAsBytes(this.item));
                }
                catch (Exception e1) {
                    logger.error("failed to process item {" + this.item.toString() + "}", (Throwable)e1);
                }
            }
        }

        private void flagPush() {
            this.pushback = true;
        }

        private boolean isPushBack() {
            return this.pushback;
        }

        @Override
        public void run() {
            try {
                ((QueueProcessor)this.queueProcessor).tryItem(this.item);
                if (this.consumer.consume(this.item) == Consumer.Result.FAIL_REQUEUE) {
                    this.flagPush();
                }
            }
            catch (InterruptedException e) {
                this.flagPush();
                Thread.currentThread().interrupt();
            }
            catch (Exception e) {
                logger.error("failed to process item {" + this.item.toString() + "}", (Throwable)e);
            }
            finally {
                this.pushBackIfNeeded();
            }
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null) {
                return false;
            }
            if (this.getClass() != o.getClass()) {
                return false;
            }
            ProcessItem p = (ProcessItem)o;
            return Objects.equals(this.item.toString(), p.item.toString());
        }

        public int hashCode() {
            return this.item.toString().hashCode();
        }
    }

    public static class Builder {
        private Path queuePath;
        private String queueName;
        private Class type;
        private int maxTries = 0;
        private int retryDelay = 1;
        private int persistRetryDelay = 0;
        private int maxRetryDelay = 1;
        private TimeUnit retryDelayUnit = TimeUnit.SECONDS;
        private TimeUnit persistRetryDelayUnit = TimeUnit.SECONDS;
        private Consumer consumer;
        private Expiration expiration;
        private RetryDelayAlgorithm retryDelayAlgorithm = RetryDelayAlgorithm.FIXED;

        public Builder() {
        }

        public Builder(String queueName, Path queuePath, Class type, Consumer consumer) throws IllegalArgumentException {
            if (queueName == null) {
                throw new IllegalArgumentException("queue name must be specified");
            }
            if (queuePath == null) {
                throw new IllegalArgumentException("queue path must be specified");
            }
            if (type == null) {
                throw new IllegalArgumentException("item type must be specified");
            }
            if (consumer == null) {
                throw new IllegalArgumentException("consumer must be specified");
            }
            this.queueName = queueName;
            this.queuePath = queuePath;
            this.type = type;
            this.consumer = consumer;
        }

        public Builder queuePath(Path queuePath) {
            this.queuePath = queuePath;
            return this;
        }

        public Path getQueuePath() {
            return this.queuePath;
        }

        public Builder queueName(String queueName) {
            this.queueName = queueName;
            return this;
        }

        public String getQueueName() {
            return this.queueName;
        }

        public Builder type(Class type) {
            this.type = type;
            return this;
        }

        public Class getType() {
            return this.type;
        }

        public Builder maxTries(int maxTries) {
            this.maxTries = maxTries;
            return this;
        }

        public int getMaxTries() {
            return this.maxTries;
        }

        public Builder retryDelay(int retryDelay) {
            this.retryDelay = retryDelay;
            return this;
        }

        public int getRetryDelay() {
            return this.retryDelay;
        }

        public Builder maxRetryDelay(int maxRetryDelay) {
            this.maxRetryDelay = maxRetryDelay;
            return this;
        }

        public int getMaxRetryDelay() {
            return this.maxRetryDelay;
        }

        public Builder persistRetryDelay(int persistRetryDelay) {
            this.persistRetryDelay = persistRetryDelay;
            return this;
        }

        public int getPersistRetryDelay() {
            return this.persistRetryDelay;
        }

        public Builder persistRetryDelayUnit(TimeUnit persistRetryDelayUnit) {
            this.persistRetryDelayUnit = persistRetryDelayUnit;
            return this;
        }

        public TimeUnit getPersistRetryDelayUnit() {
            return this.persistRetryDelayUnit;
        }

        public Builder retryDelayUnit(TimeUnit retryDelayUnit) {
            this.retryDelayUnit = retryDelayUnit;
            return this;
        }

        public TimeUnit getRetryDelayUnit() {
            return this.retryDelayUnit;
        }

        public Builder retryDelayAlgorithm(RetryDelayAlgorithm retryDelayAlgorithm) {
            this.retryDelayAlgorithm = retryDelayAlgorithm;
            return this;
        }

        public RetryDelayAlgorithm getRetryDelayAlgorithm() {
            return this.retryDelayAlgorithm;
        }

        public Builder consumer(Consumer consumer) {
            this.consumer = consumer;
            return this;
        }

        public Consumer getConsumer() {
            return this.consumer;
        }

        public Builder expiration(Expiration expiration) {
            this.expiration = expiration;
            return this;
        }

        public Expiration getExpiration() {
            return this.expiration;
        }

        public QueueProcessor build() throws IOException, IllegalStateException, IllegalArgumentException {
            return new QueueProcessor(this);
        }
    }

    public static enum RetryDelayAlgorithm {
        FIXED,
        EXPONENTIAL;

    }
}

