/*
 * Decompiled with CFR 0.152.
 */
package com.thecoderscorner.menu.remote.rs232;

import com.fazecast.jSerialComm.SerialPort;
import com.thecoderscorner.menu.remote.ConnectionChangeListener;
import com.thecoderscorner.menu.remote.MenuCommandProtocol;
import com.thecoderscorner.menu.remote.RemoteConnector;
import com.thecoderscorner.menu.remote.RemoteConnectorListener;
import com.thecoderscorner.menu.remote.commands.MenuCommand;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Rs232RemoteConnector
implements RemoteConnector {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final List<RemoteConnectorListener> connectorListeners = new CopyOnWriteArrayList<RemoteConnectorListener>();
    private final List<ConnectionChangeListener> connectionListeners = new CopyOnWriteArrayList<ConnectionChangeListener>();
    private final MenuCommandProtocol protocol;
    private final ScheduledExecutorService executor;
    private final String portName;
    private final SerialPort serialPort;
    private final byte[] cmdData = new byte[2048];
    private final ByteBuffer cmdBuffer = ByteBuffer.wrap(this.cmdData);
    private final AtomicReference<Rs232State> state = new AtomicReference<Rs232State>(Rs232State.STARTED);

    public Rs232RemoteConnector(String portName, int baud, MenuCommandProtocol protocol, ScheduledExecutorService executor) {
        this.serialPort = SerialPort.getCommPort((String)portName);
        this.serialPort.setBaudRate(baud);
        this.protocol = protocol;
        this.portName = portName;
        this.executor = executor;
    }

    @Override
    public void start() {
        this.executor.execute(this::threadedReader);
    }

    @Override
    public void stop() {
        this.executor.shutdownNow();
    }

    private void threadedReader() {
        this.logger.info("RS232 Reading thread started");
        while (!Thread.currentThread().isInterrupted()) {
            if (!this.reconnectWithWait()) continue;
            this.processMessagesOnConnection();
        }
        this.logger.info("RS232 Reading thread ended with interrupted state");
    }

    private void processMessagesOnConnection() {
        try {
            this.logger.info("Connected to serial port " + this.portName);
            this.state.set(Rs232State.CONNECTED);
            this.notifyConnection();
            while (!Thread.currentThread().isInterrupted() && this.isConnected()) {
                String line = this.readLineFromStream(this.serialPort.getInputStream());
                this.logger.debug("Line read from stream: {}", (Object)line);
                MenuCommand mc = this.protocol.fromChannel(ByteBuffer.wrap(line.getBytes()));
                this.notifyListeners(mc);
            }
            this.logger.info("Disconnected from serial port " + this.portName);
        }
        catch (Exception e) {
            this.logger.error("Disconnected from serial port " + this.portName, (Throwable)e);
        }
        finally {
            this.close();
        }
    }

    private void notifyListeners(MenuCommand mc) {
        this.connectorListeners.forEach(listener -> listener.onCommand(this, mc));
    }

    private void notifyConnection() {
        this.connectionListeners.forEach(listener -> listener.connectionChange(this, this.isConnected()));
    }

    private boolean reconnectWithWait() {
        try {
            Thread.sleep(500L);
            this.logger.info("Attempting to connect over rs232");
            this.serialPort.openPort();
            this.serialPort.setComPortTimeouts(256, 100, 0);
            if (this.serialPort.isOpen()) {
                this.notifyConnection();
            } else {
                Thread.sleep(5000L);
            }
            return this.serialPort.isOpen();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void sendMenuCommand(MenuCommand msg) throws IOException {
        if (this.isConnected()) {
            ByteBuffer byteBuffer = this.cmdBuffer;
            synchronized (byteBuffer) {
                this.cmdBuffer.clear();
                this.protocol.toChannel(this.cmdBuffer, msg);
                this.cmdBuffer.flip();
                this.logger.debug("Sending message on rs232: " + this.cmdBuffer);
                this.serialPort.getOutputStream().write(96);
                this.serialPort.getOutputStream().write(this.cmdData, 0, this.cmdBuffer.remaining());
            }
        } else {
            throw new IOException("Not connected to port");
        }
    }

    @Override
    public boolean isConnected() {
        return this.state.get() == Rs232State.CONNECTED;
    }

    @Override
    public String getConnectionName() {
        return this.portName;
    }

    @Override
    public void registerConnectorListener(RemoteConnectorListener listener) {
        this.connectorListeners.add(listener);
    }

    @Override
    public void registerConnectionChangeListener(ConnectionChangeListener listener) {
        this.connectionListeners.add(listener);
        this.executor.execute(() -> listener.connectionChange(this, this.isConnected()));
    }

    @Override
    public void close() {
        this.logger.info("Closing rs232 port");
        this.state.set(Rs232State.DISCONNECTED);
        this.notifyConnection();
    }

    private String readLineFromStream(InputStream stream) throws IOException {
        if (stream == null) {
            this.close();
            throw new IOException("The connection closed on unexpectedly - stream was null");
        }
        StringBuilder sb = new StringBuilder(100);
        boolean backtickFound = false;
        while (!backtickFound) {
            int readByte = stream.read();
            backtickFound = readByte == 96;
        }
        boolean crFound = false;
        while (!crFound) {
            int read = stream.read();
            boolean bl = crFound = read == 10 || read == 13;
            if (read < 0) continue;
            sb.append((char)read);
        }
        return sb.toString();
    }

    static enum Rs232State {
        STARTED,
        CONNECTED,
        DISCONNECTED;

    }
}

