/**
 * The MIT License (MIT)
 * <p>
 * Copyright (c) 2014 the original author or authors.
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.wandrell.pattern.command;

/**
 * Interface for the command executor pattern. It is designed to work with the
 * command pattern interfaces existing on this same package: {@link Command} ,
 * {@link ResultCommand} and {@link UndoableCommand}.
 * <p>
 * It is the executor's main job to receive one of these commands through its
 * own {@link #execute(Command) execute} method and processing it by calling
 * that same command's {@code execute} method, differentiating between the bare
 * {@code Command}, which returns nothing, and the {@code ResultCommand}, which
 * generates a value during it's execution.
 * <p>
 * For the first a {@link #execute(Command) execute(Command)}, with a
 * {@code void} type, is used. While the second is executed through the
 * {@link #execute(ResultCommand) execute(ResultCommand)} which will return the
 * value generated by the command.
 * <p>
 * Both methods are expected to execute the command as soon as they receive it.
 * There can be additional operations, such as catching exceptions, injecting
 * dependencies or setting up the environment. But these should be minimal, and
 * never postpone the command's execution more than necessary.
 * <p>
 * Additionally, implementations of {@link UndoableCommand} are handled through
 * the {@link #undo(UndoableCommand) undo} method, which will take care of the
 * command's own {@code undo} method in a similar way to the previously
 * explained {@code execute} method.
 * 
 * @author Bernardo Martínez Garrido
 * @see Command
 * @see ResultCommand
 * @see UndoableCommand
 */
public interface CommandExecutor {

    /**
     * Executes the received {@code Command}.
     * <p>
     * Any exception thrown by the command is expected to be caught and handled
     * inside this method.
     * 
     * @param command
     *            the {@code Command} to be executed
     */
    public void execute(final Command command);

    /**
     * Executes the received {@code ResultCommand} and returns the value
     * generated by this operation.
     * <p>
     * Any exception thrown by the command is expected to be caught and handled
     * inside this method.
     * 
     * @param command
     *            the {@code ResultCommand} to be executed
     * @param <V>
     *            the type generated by the command
     * @return an object generated by the command
     */
    public <V> V execute(final ResultCommand<V> command);

    /**
     * Undoes the received {@code UndoableCommand}.
     * <p>
     * If there is nothing to undo then the method is expected to act as if it
     * ignored the command, doing nothing.
     * <p>
     * Any exception thrown by the command is expected to be caught and handled
     * inside this method.
     * 
     * @param command
     *            the {@code UndoableCommand} to be undone
     */
    public void undo(final UndoableCommand command);

}
