/*
 * Copyright (c) 2020 Connor Goulding
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.bindingz.api.client.jackson;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.kjetland.jackson.jsonSchema.JsonSchemaConfig;
import com.kjetland.jackson.jsonSchema.JsonSchemaDraft;
import com.kjetland.jackson.jsonSchema.JsonSchemaGenerator;
import io.bindingz.api.annotations.jackson.JacksonConfiguration;
import io.bindingz.api.annotations.jackson.ConfigurationFactory;
import io.bindingz.api.client.SchemaService;
import io.bindingz.api.model.JsonSchemaSpec;
import org.reflections.Reflections;
import org.reflections.scanners.SubTypesScanner;
import org.reflections.util.ConfigurationBuilder;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.stream.Collectors;

public class JacksonSchemaService implements SchemaService {

    private final JsonSchemaGenerator generator;

    public JacksonSchemaService(List<ClassLoader> classLoaders) {
        this.generator = createDefaultGenerator(classLoaders);
    }

    public JacksonSchemaService(List<ClassLoader> classLoaders, JacksonConfiguration configuration) {
        if (!configuration.factory().equals(ConfigurationFactory.class)) {
            try {
                ConfigurationFactory factory = configuration.factory().getConstructor().newInstance();
                generator = new JsonSchemaGenerator(
                        factory.createObjectMapper(),
                        createConfig(factory.customTypeMappings())
                );
            } catch (Exception e) {
                throw new IllegalArgumentException("Unable to create ConfigurationFactory");
            }
        } else {
            generator = createDefaultGenerator(classLoaders);
        }
    }

    @Override
    public Map<JsonSchemaSpec, JsonNode> createSchemas(Class contract) {
        Map<JsonSchemaSpec, JsonNode> schemas = new HashMap<>();
        schemas.put(JsonSchemaSpec.DRAFT_04, generator.generateJsonSchema(contract));
        return schemas;
    }

    private JsonSchemaGenerator createDefaultGenerator(List<ClassLoader> classLoaders) {
        Map<String, String> typeMappings = new HashMap<>();
        typeMappings.put("java.time.LocalDateTime", "datetime-local");
        typeMappings.put("java.time.OffsetDateTime", "datetime");
        typeMappings.put("java.time.LocalDate", "date");
        typeMappings.put("org.joda.time.LocalDate", "date");
        return new JsonSchemaGenerator(
                createDefaultObjectMapper(classLoaders),
                createConfig(typeMappings)
        );
    }

    private JsonSchemaConfig createConfig(Map<String, String> customTypesToFormats) {
        JsonSchemaConfig config = JsonSchemaConfig.create(
                false,
                Optional.empty(),
                true,
                true,
                false,
                false,
                false,
                false,
                false,
                customTypesToFormats,
                false,
                new HashSet<>(Arrays.asList(
                        scala.collection.immutable.Set.class,
                        scala.collection.mutable.Set.class,
                        java.util.Set.class
                )),
                Collections.emptyMap(),
                Collections.emptyMap(),
                null,
                true,
                null
        );

        return config.withJsonSchemaDraft(version(JsonSchemaSpec.DRAFT_04));
    }

    private ObjectMapper createDefaultObjectMapper(List<ClassLoader> classLoaders) {
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);

        String[] packages = Arrays.stream(Package.getPackages())
                .map(aPackage -> aPackage.getName().split("\\.")[0])
                .collect(Collectors.toSet())
                .toArray(new String[]{});

        Reflections reflections = new Reflections(ConfigurationBuilder.build()
                .addClassLoaders(classLoaders)
                .forPackages(packages)
                .addScanners(new SubTypesScanner())
        );

        Set<Class<? extends Module>> modules = reflections.getSubTypesOf(Module.class);
        for (Class<? extends Module> moduleClass : modules) {
            try {
                Constructor[] constructors = moduleClass.getConstructors();
                for (Constructor<Module> constructor : constructors) {
                    if (constructor.getParameterCount() == 0) {
                        Module module = constructor.newInstance();
                        mapper.registerModule(module);
                        System.out.println("Registered module " + moduleClass);
                    }
                }
            } catch (IllegalAccessException | InstantiationException | InvocationTargetException e) {
                e.printStackTrace();
            }
        }

        return mapper;
    }

    private JsonSchemaDraft version(JsonSchemaSpec spec) {
        return JsonSchemaDraft.valueOf(spec.name());
    }
}
