/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for an embedded transaction header
**/
public class EmbeddedTransactionHeaderBuilder implements Serializer {

    /** Entity size. **/
    private int size;

    /** Reserved padding to align end of EmbeddedTransactionHeader on 8-byte boundary. **/
    private final int embeddedTransactionHeader_Reserved1;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected EmbeddedTransactionHeaderBuilder(DataInputStream stream) {
        try {
            this.size = Integer.reverseBytes(stream.readInt());
            this.embeddedTransactionHeader_Reserved1 = Integer.reverseBytes(stream.readInt());
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of EmbeddedTransactionHeaderBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of EmbeddedTransactionHeaderBuilder.
     */
    public static EmbeddedTransactionHeaderBuilder loadFromBinary(DataInputStream stream) {
        return new EmbeddedTransactionHeaderBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    */
    protected EmbeddedTransactionHeaderBuilder() {
        this.embeddedTransactionHeader_Reserved1 = 0;
    }
    
    /**
     * Creates an instance of EmbeddedTransactionHeaderBuilder.
     *
     * @return Instance of EmbeddedTransactionHeaderBuilder.
     */
    public static EmbeddedTransactionHeaderBuilder create() {
        return new EmbeddedTransactionHeaderBuilder();
    }

    /**
     * Gets entity size.
     *
     * @return Entity size.
     */
    public int getStreamSize() {
        return this.size;
    }

    /**
     * Gets reserved padding to align end of EmbeddedTransactionHeader on 8-byte boundary.
     *
     * @return Reserved padding to align end of EmbeddedTransactionHeader on 8-byte boundary.
     */
    public int getEmbeddedTransactionHeader_Reserved1() {
        return this.embeddedTransactionHeader_Reserved1;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 4;
        size += 4;
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getSize()));
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getEmbeddedTransactionHeader_Reserved1()));
        });
    }
}

