/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a multisig entry
**/
public class MultisigEntryBuilder implements Serializer {

    /** Minimum approval for modifications. **/
    private final int minApproval;

    /** Minimum approval for removal. **/
    private final int minRemoval;

    /** Account public key. **/
    private final KeyDto accountPublicKey;

    /** Cosignatories for account. **/
    private final List<KeyDto> cosignatoryPublicKeys;

    /** Accounts for which the entry is cosignatory. **/
    private final List<KeyDto> multisigPublicKeys;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MultisigEntryBuilder(DataInputStream stream) {
        try {
            this.minApproval = Integer.reverseBytes(stream.readInt());
            this.minRemoval = Integer.reverseBytes(stream.readInt());
            this.accountPublicKey = KeyDto.loadFromBinary(stream);
            final long cosignatoryPublicKeysCount = Long.reverseBytes(stream.readLong());
            this.cosignatoryPublicKeys = GeneratorUtils.loadFromBinaryArray(KeyDto::loadFromBinary, stream, cosignatoryPublicKeysCount);
            final long multisigPublicKeysCount = Long.reverseBytes(stream.readLong());
            this.multisigPublicKeys = GeneratorUtils.loadFromBinaryArray(KeyDto::loadFromBinary, stream, multisigPublicKeysCount);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MultisigEntryBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MultisigEntryBuilder.
     */
    public static MultisigEntryBuilder loadFromBinary(DataInputStream stream) {
        return new MultisigEntryBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param minApproval Minimum approval for modifications.
    * @param minRemoval Minimum approval for removal.
    * @param accountPublicKey Account public key.
    * @param cosignatoryPublicKeys Cosignatories for account.
    * @param multisigPublicKeys Accounts for which the entry is cosignatory.
    */
    protected MultisigEntryBuilder(int minApproval, int minRemoval, KeyDto accountPublicKey, List<KeyDto> cosignatoryPublicKeys, List<KeyDto> multisigPublicKeys) {
        GeneratorUtils.notNull(minApproval, "minApproval is null");
        GeneratorUtils.notNull(minRemoval, "minRemoval is null");
        GeneratorUtils.notNull(accountPublicKey, "accountPublicKey is null");
        GeneratorUtils.notNull(cosignatoryPublicKeys, "cosignatoryPublicKeys is null");
        GeneratorUtils.notNull(multisigPublicKeys, "multisigPublicKeys is null");
        this.minApproval = minApproval;
        this.minRemoval = minRemoval;
        this.accountPublicKey = accountPublicKey;
        this.cosignatoryPublicKeys = cosignatoryPublicKeys;
        this.multisigPublicKeys = multisigPublicKeys;
    }
    
    /**
     * Creates an instance of MultisigEntryBuilder.
     *
     * @param minApproval Minimum approval for modifications.
     * @param minRemoval Minimum approval for removal.
     * @param accountPublicKey Account public key.
     * @param cosignatoryPublicKeys Cosignatories for account.
     * @param multisigPublicKeys Accounts for which the entry is cosignatory.
     * @return Instance of MultisigEntryBuilder.
     */
    public static MultisigEntryBuilder create(int minApproval, int minRemoval, KeyDto accountPublicKey, List<KeyDto> cosignatoryPublicKeys, List<KeyDto> multisigPublicKeys) {
        return new MultisigEntryBuilder(minApproval, minRemoval, accountPublicKey, cosignatoryPublicKeys, multisigPublicKeys);
    }

    /**
     * Gets minimum approval for modifications.
     *
     * @return Minimum approval for modifications.
     */
    public int getMinApproval() {
        return this.minApproval;
    }

    /**
     * Gets minimum approval for removal.
     *
     * @return Minimum approval for removal.
     */
    public int getMinRemoval() {
        return this.minRemoval;
    }

    /**
     * Gets account public key.
     *
     * @return Account public key.
     */
    public KeyDto getAccountPublicKey() {
        return this.accountPublicKey;
    }

    /**
     * Gets cosignatories for account.
     *
     * @return Cosignatories for account.
     */
    public List<KeyDto> getCosignatoryPublicKeys() {
        return this.cosignatoryPublicKeys;
    }

    /**
     * Gets accounts for which the entry is cosignatory.
     *
     * @return Accounts for which the entry is cosignatory.
     */
    public List<KeyDto> getMultisigPublicKeys() {
        return this.multisigPublicKeys;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 4;
        size += 4;
        size += this.accountPublicKey.getSize();
        size += 8;
        size += this.cosignatoryPublicKeys.stream().mapToInt(o -> o.getSize()).sum();
        size += 8;
        size += this.multisigPublicKeys.stream().mapToInt(o -> o.getSize()).sum();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getMinApproval()));
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getMinRemoval()));
            GeneratorUtils.writeEntity(dataOutputStream, this.accountPublicKey);
            dataOutputStream.writeLong(Long.reverseBytes((long) GeneratorUtils.getSize(this.getCosignatoryPublicKeys())));
            GeneratorUtils.writeList(dataOutputStream, this.cosignatoryPublicKeys);
            dataOutputStream.writeLong(Long.reverseBytes((long) GeneratorUtils.getSize(this.getMultisigPublicKeys())));
            GeneratorUtils.writeList(dataOutputStream, this.multisigPublicKeys);
        });
    }
}

