/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;

/** Binary layout for a balance transfer receipt. */
public final class BalanceTransferReceiptBuilder extends ReceiptBuilder implements Serializer {
    /** Mosaic. */
    private final MosaicBuilder mosaic;
    /** Mosaic sender public key. */
    private final KeyDto senderPublicKey;
    /** Mosaic recipient address. */
    private final AddressDto recipientAddress;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected BalanceTransferReceiptBuilder(final DataInputStream stream) {
        super(stream);
        this.mosaic = MosaicBuilder.loadFromBinary(stream);
        this.senderPublicKey = KeyDto.loadFromBinary(stream);
        this.recipientAddress = AddressDto.loadFromBinary(stream);
    }

    /**
     * Constructor.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param mosaic Mosaic.
     * @param senderPublicKey Mosaic sender public key.
     * @param recipientAddress Mosaic recipient address.
     */
    protected BalanceTransferReceiptBuilder(final short version, final ReceiptTypeDto type, final MosaicBuilder mosaic, final KeyDto senderPublicKey, final AddressDto recipientAddress) {
        super(version, type);
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        GeneratorUtils.notNull(senderPublicKey, "senderPublicKey is null");
        GeneratorUtils.notNull(recipientAddress, "recipientAddress is null");
        this.mosaic = mosaic;
        this.senderPublicKey = senderPublicKey;
        this.recipientAddress = recipientAddress;
    }

    /**
     * Creates an instance of BalanceTransferReceiptBuilder.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param mosaic Mosaic.
     * @param senderPublicKey Mosaic sender public key.
     * @param recipientAddress Mosaic recipient address.
     * @return Instance of BalanceTransferReceiptBuilder.
     */
    public static BalanceTransferReceiptBuilder create(final short version, final ReceiptTypeDto type, final MosaicBuilder mosaic, final KeyDto senderPublicKey, final AddressDto recipientAddress) {
        return new BalanceTransferReceiptBuilder(version, type, mosaic, senderPublicKey, recipientAddress);
    }

    /**
     * Gets mosaic.
     *
     * @return Mosaic.
     */
    public MosaicBuilder getMosaic() {
        return this.mosaic;
    }

    /**
     * Gets mosaic sender public key.
     *
     * @return Mosaic sender public key.
     */
    public KeyDto getSenderPublicKey() {
        return this.senderPublicKey;
    }

    /**
     * Gets mosaic recipient address.
     *
     * @return Mosaic recipient address.
     */
    public AddressDto getRecipientAddress() {
        return this.recipientAddress;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    @Override
    public int getSize() {
        int size = super.getSize();
        size += this.mosaic.getSize();
        size += this.senderPublicKey.getSize();
        size += this.recipientAddress.getSize();
        return size;
    }

    /**
     * Creates an instance of BalanceTransferReceiptBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of BalanceTransferReceiptBuilder.
     */
    public static BalanceTransferReceiptBuilder loadFromBinary(final DataInputStream stream) {
        return new BalanceTransferReceiptBuilder(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            final byte[] mosaicBytes = this.mosaic.serialize();
            dataOutputStream.write(mosaicBytes, 0, mosaicBytes.length);
            final byte[] senderPublicKeyBytes = this.senderPublicKey.serialize();
            dataOutputStream.write(senderPublicKeyBytes, 0, senderPublicKeyBytes.length);
            final byte[] recipientAddressBytes = this.recipientAddress.serialize();
            dataOutputStream.write(recipientAddressBytes, 0, recipientAddressBytes.length);
        });
    }
}
