/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;

/** Enumeration of entity types. */
public enum EntityTypeDto {
    /** Reserved entity type. */
    RESERVED((short) 0),
    /** Account link transaction builder. */
    ACCOUNT_LINK_TRANSACTION_BUILDER((short) 16716),
    /** Aggregate complete transaction builder. */
    AGGREGATE_COMPLETE_TRANSACTION_BUILDER((short) 16705),
    /** Aggregate bonded transaction builder. */
    AGGREGATE_BONDED_TRANSACTION_BUILDER((short) 16961),
    /** Hash lock transaction builder. */
    HASH_LOCK_TRANSACTION_BUILDER((short) 16712),
    /** Secret lock transaction builder. */
    SECRET_LOCK_TRANSACTION_BUILDER((short) 16722),
    /** Secret proof transaction builder. */
    SECRET_PROOF_TRANSACTION_BUILDER((short) 16978),
    /** Account metadata transaction builder. */
    ACCOUNT_METADATA_TRANSACTION_BUILDER((short) 16708),
    /** Mosaic metadata transaction builder. */
    MOSAIC_METADATA_TRANSACTION_BUILDER((short) 16964),
    /** Namespace metadata transaction builder. */
    NAMESPACE_METADATA_TRANSACTION_BUILDER((short) 17220),
    /** Mosaic definition transaction builder. */
    MOSAIC_DEFINITION_TRANSACTION_BUILDER((short) 16717),
    /** Mosaic supply change transaction builder. */
    MOSAIC_SUPPLY_CHANGE_TRANSACTION_BUILDER((short) 16973),
    /** Multisig account modification transaction builder. */
    MULTISIG_ACCOUNT_MODIFICATION_TRANSACTION_BUILDER((short) 16725),
    /** Address alias transaction builder. */
    ADDRESS_ALIAS_TRANSACTION_BUILDER((short) 16974),
    /** Mosaic alias transaction builder. */
    MOSAIC_ALIAS_TRANSACTION_BUILDER((short) 17230),
    /** Namespace registration transaction builder. */
    NAMESPACE_REGISTRATION_TRANSACTION_BUILDER((short) 16718),
    /** Account address restriction transaction builder. */
    ACCOUNT_ADDRESS_RESTRICTION_TRANSACTION_BUILDER((short) 16720),
    /** Account mosaic restriction transaction builder. */
    ACCOUNT_MOSAIC_RESTRICTION_TRANSACTION_BUILDER((short) 16976),
    /** Account operation restriction transaction builder. */
    ACCOUNT_OPERATION_RESTRICTION_TRANSACTION_BUILDER((short) 17232),
    /** Mosaic address restriction transaction builder. */
    MOSAIC_ADDRESS_RESTRICTION_TRANSACTION_BUILDER((short) 16977),
    /** Mosaic global restriction transaction builder. */
    MOSAIC_GLOBAL_RESTRICTION_TRANSACTION_BUILDER((short) 16721),
    /** Transfer transaction builder. */
    TRANSFER_TRANSACTION_BUILDER((short) 16724);

    /**
     * Gets the value of the enum.
     *
     * @return Value of the enum.
     */
    public short getValue() {
        return this.value;
    }

    /** Enum value. */
    private final short value;

    /**
     * Constructor.
     *
     * @param value Enum value.
     */
     EntityTypeDto(final short value) {
        this.value = value;
    }

    /**
     * Gets enum value.
     *
     * @param value Raw value of the enum.
     * @return Enum value.
     */
    public static EntityTypeDto rawValueOf(final short value) {
        for (EntityTypeDto current : EntityTypeDto.values()) {
            if (value == current.value) {
                return current;
            }
        }
        throw new IllegalArgumentException(value + " was not a backing value for EntityTypeDto.");
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        return 2;
    }

    /**
     * Creates an instance of EntityTypeDto from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of EntityTypeDto.
     */
    public static EntityTypeDto loadFromBinary(final DataInputStream stream) {
        try {
            final short streamValue = Short.reverseBytes(stream.readShort());
            return rawValueOf(streamValue);
        } catch(Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            dataOutputStream.writeShort(Short.reverseBytes(this.value));
        });
    }
}
