/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;

/** Binary layout for mosaic resolution entry. */
public class MosaicResolutionEntryBuilder implements Serializer {
    /** Source of resolution within block. */
    private final ReceiptSourceBuilder source;
    /** Resolved value. */
    private final MosaicIdDto resolved;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicResolutionEntryBuilder(final DataInputStream stream) {
        this.source = ReceiptSourceBuilder.loadFromBinary(stream);
        this.resolved = MosaicIdDto.loadFromBinary(stream);
    }

    /**
     * Constructor.
     *
     * @param source Source of resolution within block.
     * @param resolved Resolved value.
     */
    protected MosaicResolutionEntryBuilder(final ReceiptSourceBuilder source, final MosaicIdDto resolved) {
        GeneratorUtils.notNull(source, "source is null");
        GeneratorUtils.notNull(resolved, "resolved is null");
        this.source = source;
        this.resolved = resolved;
    }

    /**
     * Creates an instance of MosaicResolutionEntryBuilder.
     *
     * @param source Source of resolution within block.
     * @param resolved Resolved value.
     * @return Instance of MosaicResolutionEntryBuilder.
     */
    public static MosaicResolutionEntryBuilder create(final ReceiptSourceBuilder source, final MosaicIdDto resolved) {
        return new MosaicResolutionEntryBuilder(source, resolved);
    }

    /**
     * Gets source of resolution within block.
     *
     * @return Source of resolution within block.
     */
    public ReceiptSourceBuilder getSource() {
        return this.source;
    }

    /**
     * Gets resolved value.
     *
     * @return Resolved value.
     */
    public MosaicIdDto getResolved() {
        return this.resolved;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.source.getSize();
        size += this.resolved.getSize();
        return size;
    }

    /**
     * Creates an instance of MosaicResolutionEntryBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicResolutionEntryBuilder.
     */
    public static MosaicResolutionEntryBuilder loadFromBinary(final DataInputStream stream) {
        return new MosaicResolutionEntryBuilder(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            final byte[] sourceBytes = this.source.serialize();
            dataOutputStream.write(sourceBytes, 0, sourceBytes.length);
            final byte[] resolvedBytes = this.resolved.serialize();
            dataOutputStream.write(resolvedBytes, 0, resolvedBytes.length);
        });
    }
}
