/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;

/** Binary layout for a namespace expiry receipt. */
public final class NamespaceExpiryReceiptBuilder extends ReceiptBuilder implements Serializer {
    /** Expiring namespace id. */
    private final NamespaceIdDto artifactId;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected NamespaceExpiryReceiptBuilder(final DataInputStream stream) {
        super(stream);
        this.artifactId = NamespaceIdDto.loadFromBinary(stream);
    }

    /**
     * Constructor.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param artifactId Expiring namespace id.
     */
    protected NamespaceExpiryReceiptBuilder(final short version, final ReceiptTypeDto type, final NamespaceIdDto artifactId) {
        super(version, type);
        GeneratorUtils.notNull(artifactId, "artifactId is null");
        this.artifactId = artifactId;
    }

    /**
     * Creates an instance of NamespaceExpiryReceiptBuilder.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param artifactId Expiring namespace id.
     * @return Instance of NamespaceExpiryReceiptBuilder.
     */
    public static NamespaceExpiryReceiptBuilder create(final short version, final ReceiptTypeDto type, final NamespaceIdDto artifactId) {
        return new NamespaceExpiryReceiptBuilder(version, type, artifactId);
    }

    /**
     * Gets expiring namespace id.
     *
     * @return Expiring namespace id.
     */
    public NamespaceIdDto getArtifactId() {
        return this.artifactId;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    @Override
    public int getSize() {
        int size = super.getSize();
        size += this.artifactId.getSize();
        return size;
    }

    /**
     * Creates an instance of NamespaceExpiryReceiptBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of NamespaceExpiryReceiptBuilder.
     */
    public static NamespaceExpiryReceiptBuilder loadFromBinary(final DataInputStream stream) {
        return new NamespaceExpiryReceiptBuilder(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            final byte[] artifactIdBytes = this.artifactId.serialize();
            dataOutputStream.write(artifactIdBytes, 0, artifactIdBytes.length);
        });
    }
}
