package io.split.client;

import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import io.codigo.dtos.KeyImpressionDTO;
import io.codigo.dtos.TestImpressionsDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.List;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A cache removal listener that posts the removed value to a rest endpoint.
 *
 * @author adil
 */
public final class CachedTreatmentLogRemovalListener implements RemovalListener<String, List<KeyImpressionDTO>> {
    private static final Logger _log = LoggerFactory.getLogger(CachedTreatmentLogRemovalListener.class);

    private final WebTarget _target;

    public static CachedTreatmentLogRemovalListener create(WebTarget root) {
        return new CachedTreatmentLogRemovalListener(root.path("testImpressions"));
    }

    private CachedTreatmentLogRemovalListener(WebTarget target) {
        _target = target;

        checkNotNull(_target);
    }


    @Override
    public void onRemoval(RemovalNotification<String, List<KeyImpressionDTO>> removalNotification) {
        String test = removalNotification.getKey();
        List<KeyImpressionDTO> keys = removalNotification.getValue();

        if (keys.isEmpty()) {
            _log.info("No impressions to report: " + test);
            return;
        }

        Set<String> keysSeen = Sets.newHashSet();
        List<KeyImpressionDTO> filtered = Lists.newArrayList();

        for (int i = (keys.size() - 1); i >= 0; i--) {
            KeyImpressionDTO keyImpressionDTO = keys.get(i);
            if (keysSeen.contains(keyImpressionDTO.keyName())) {
                continue;
            }

            keysSeen.add(keyImpressionDTO.keyName());
            filtered.add(keyImpressionDTO);
        }

        if (filtered.isEmpty()) {
            _log.info("No impressions to report post filtering: " + test);
            return;
        }

        if (_log.isDebugEnabled()) {
            _log.debug("Filteration: " + filtered.size() + " left from " + keys.size());
        }

        TestImpressionsDTO testImpressionsDTO = TestImpressionsDTO.builder()
                .testName(test)
                .keyImpressionsDTO(filtered)
                .build();


        try {
            Response response = _target
                    .request(MediaType.APPLICATION_JSON)
                    .post(Entity.json(testImpressionsDTO));

            if (response.getStatus() != 200) {
                _log.warn("Response status was: " + response.getStatus());
            }

        } catch (Exception e) {
            _log.error("Exception when posting impressions for " + test, e);
        }


    }
}
