package io.split.client;

import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * For environment 'localhost', this class reads the list of partitions that should be 100% on
 * from a file $HOME/.partitions, and returns an instance of TheseSplitsAre100PercentOnSplitClient.
 * <p>
 *
 * The format of the $HOME/.partitions is one split name per line. Leading and trailing whitespaces
 * will be ignored. Lines that start with '#' will be ignored too as comments. All partitions
 * specified in the file will be 100% on for all users; every other split will be 100% off
 * for all users.
 * <p>
 *
 * If $HOME/.partitions is not found, all partitions are assumed off.
 *
 * @author adil
 */
public class LocalhostSplitClientBuilder {

    private static final Logger _log = LoggerFactory.getLogger(LocalhostSplitClientBuilder.class);

    public static final String LOCALHOST = "localhost";
    public static final String FILENAME = ".split";

    public static TheseFeaturesAreOnSplitClient build() throws IOException {
        String home = System.getProperty("user.home");
        checkNotNull(home, "Property user.home should be set when using environment: " + LOCALHOST);

        return build(home);
    }

    static TheseFeaturesAreOnSplitClient build(String home) throws IOException {
        _log.info("home = " + home);

        BufferedReader reader = null;

        String fileName = home + "/" + FILENAME;

        Set<String> onSplits = Sets.newHashSet();

        try {
            reader = new BufferedReader(new FileReader(new File(fileName)));

            for (String line = reader.readLine(); line != null; line = reader.readLine()) {
                line = line.trim();
                if (line.isEmpty() || line.startsWith("#")) {
                    continue;
                }

                if (onSplits.add(line)) {
                    _log.info("Split: " + line + " will be on for 100% of your users");
                }

            }
        } catch (FileNotFoundException e) {
            _log.warn("There was no file named " + fileName + " found. " +
                    "We created a split client that considers all partitions to be off for all of your users. " +
                    "If you wish to turn on a split, enter the name of that split in " + fileName +
                    "; one split per line. Whitelines are fine. Lines starting with '#' are considered comments");
        } catch (IOException e) {
            throw e;
        } finally {
            if (reader != null) {
                reader.close();
            }
        }

        return new TheseFeaturesAreOnSplitClient(onSplits);
    }

}
