package io.split.client;


/**
 * Configurations for the SplitClient.
 *
 * @author adil
 */
public class SplitClientConfig {

    public static final String SPLIT_SDK_VERSION = "java-0.0.2";

    private final String _endpoint;
    private final int _pollForFeatureChangesEveryNSeconds;
    private final int _pollForSegmentChangesEveryNSeconds;
    private final int _treatmentLogCacheExpireNSecondsAfterWrite;
    private final int _connectionTimeoutInMs;
    private final int _readTimeoutInMs;


    public static Builder builder() {
        return new Builder();
    }

    private SplitClientConfig(String endpoint,
                              int pollForFeatureChangesEveryNSeconds,
                              int pollForSegmentChangesEveryNSeconds,
                              int treatmentLogCacheExpireNSecondsAfterWrite,
                              int connectionTimeoutInMs,
                              int readTimeoutInMs) {
        _endpoint = endpoint;
        _pollForFeatureChangesEveryNSeconds = pollForFeatureChangesEveryNSeconds;
        _pollForSegmentChangesEveryNSeconds = pollForSegmentChangesEveryNSeconds;
        _treatmentLogCacheExpireNSecondsAfterWrite = treatmentLogCacheExpireNSecondsAfterWrite;
        _connectionTimeoutInMs = connectionTimeoutInMs;
        _readTimeoutInMs = readTimeoutInMs;
    }

    public String endpoint() {
        return _endpoint;
    }

    public int pollForFeatureChangesEveryNSeconds() {
        return _pollForFeatureChangesEveryNSeconds;
    }

    public int pollForSegmentChangesEveryNSeconds() {
        return _pollForSegmentChangesEveryNSeconds;
    }

    public int treatmentLogCacheExpireNSecondsAfterWrite() {
        return _treatmentLogCacheExpireNSecondsAfterWrite;
    }

    public int connectionTimeoutInMs() {
        return _connectionTimeoutInMs;
    }

    public int readTimeoutInMs() {
        return _readTimeoutInMs;
    }

    public static final class Builder {

        private String _endpoint = "http://sdk.split.io/api";
        private int _pollForFeatureChangesEveryNSeconds = 60;
        private int _pollForSegmentChangesEveryNSeconds = 60;
        private int _treatmentLogCacheExpireNSecondsAfterWrite = 60;
        private int _connectionTimeOutInMs = 15000;
        private int _readTimeoutInMs = 15000;
        public Builder() {}

        /**
         * The rest endpoint that sdk will hit for latest features and segments.
         *
         * @param endpoint MUST NOT be null
         *
         * @return
         */
        public Builder endpoint(String endpoint) {
            _endpoint = endpoint;
            return this;
        }

        /**
         * The SDK will poll the endpoint for changes to features at this period.
         * <p>
         * Implementation Note: The SDK actually polls at a random interval
         * chosen between (0.5 * n, n). This is to ensure that
         * SDKs that are deployed simultaneously on different machines do not
         * inundate the backend with requests at the same interval.
         * </p>
         *
         * @param n MUST be greater than 0. Default value is 60.
         * @return
         */
        public Builder pollForFeatureChangesEveryNSeconds(int n) {
            _pollForFeatureChangesEveryNSeconds = n;
            return this;
        }

        /**
         * The SDK will poll the endpoint for changes to segments at this period in seconds.
         * <p>
         * Implementation Note: The SDK actually polls at a random interval
         * chosen between (0.5 * n, n). This is to ensure that
         * SDKs that are deployed simultaneously on different machines do not
         * inundate the backend with requests at the same interval.
         * </p>
         *
         * @param n MUST be greater than 0. Default value is 60.
         * @return
         */
        public Builder pollForSegmentChangesEveryNSeconds(int n) {
            _pollForSegmentChangesEveryNSeconds = n;
            return this;
        }

        /**
         * The treatment log captures the which key saw what treatment ("on", "off")
         * at what time. This log is periodically pushed back to split endpoint.
         * This parameter controls how quickly does the cache expire after a write.
         *
         * This is an ADVANCED parameter
         *
         * @param n MUST be > 0.
         *
         * @return
         */
        public Builder treatmentLogCacheExpireNSecondsAfterWrite(int n) {
            _treatmentLogCacheExpireNSecondsAfterWrite = n;
            return this;
        }

        /**
         * Http client connection timeout. Default value is 15000ms.
         * @param ms MUST be greater than 0.
         *
         * @return
         */
        public Builder connectionTimeoutInMs(int ms) {
            _connectionTimeOutInMs = ms;
            return this;
        }

        /**
         * Http client read timeout. Default value is 15000ms.
         * @param ms MUST be greater than 0.
         *
         * @return
         */
        public Builder readTimeoutInMs(int ms) {
            _connectionTimeOutInMs = ms;
            return this;
        }


        public SplitClientConfig build() {
            if (_pollForFeatureChangesEveryNSeconds <= 0) {
                throw new IllegalArgumentException("pollForFeatureChangesEveryNSeconds must be > 0: " + _pollForFeatureChangesEveryNSeconds);
            }

            if (_pollForSegmentChangesEveryNSeconds <= 0) {
                throw new IllegalArgumentException("pollForSegmentChangesEveryNSeconds must be > 0: " + _pollForSegmentChangesEveryNSeconds);
            }

            if (_treatmentLogCacheExpireNSecondsAfterWrite <= 0) {
                throw new IllegalArgumentException("treatmentLogCacheExpireNSecondsAfterWrite must be > 0: " + _treatmentLogCacheExpireNSecondsAfterWrite);
            }

            if (_connectionTimeOutInMs <= 0) {
                throw new IllegalArgumentException("connectionTimeOutInMs must be > 0: " + _connectionTimeOutInMs);
            }

            if (_readTimeoutInMs <= 0) {
                throw new IllegalArgumentException("readTimeoutInMs must be > 0: " + _readTimeoutInMs);
            }

            if (_endpoint == null) {
                throw new IllegalArgumentException("endpoint must not be null");
            }

            return new SplitClientConfig(_endpoint,
                    _pollForFeatureChangesEveryNSeconds,
                    _pollForSegmentChangesEveryNSeconds,
                    _treatmentLogCacheExpireNSecondsAfterWrite,
                    _connectionTimeOutInMs,
                    _readTimeoutInMs);
        }

    }


}
