
package io.split.client.metrics;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import io.split.api.CounterDTO;
import io.split.api.GaugeDTO;
import io.split.api.LatencyDTO;
import io.split.engine.metrics.Metrics;

import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;


/**
 * Created by adilaijaz on 9/4/15.
 */
public class CachedMetrics implements Metrics {

    private final HttpMetrics _httpMetrics;

    private final Map<String, List<Long>> _latencyMap;
    private final Map<String, SumAndCount> _countMap;
    private final Map<String, ValueAndCount> _gaugeMap;


    private final Object _latencyLock = new Object();
    private final Object _counterLock = new Object();
    private final Object _gaugeLock = new Object();

    private final int _queueForTheseManyCalls;

    public CachedMetrics(HttpMetrics httpMetrics, int queueForTheseManyCalls) {
        _httpMetrics = httpMetrics;
        _latencyMap = Maps.newHashMap();
        _countMap = Maps.newHashMap();
        _gaugeMap = Maps.newHashMap();
        checkArgument(queueForTheseManyCalls > 0, "queue for cache should be greater than zero");
        _queueForTheseManyCalls = queueForTheseManyCalls;
    }

    @Override
    public void count(String counter, long delta) {
        if (delta <= 0) {
            return;
        }

        if (counter == null || counter.trim().isEmpty()) {
            return;
        }

        synchronized (_counterLock) {
            SumAndCount ifPresent = _countMap.get(counter);
            if (ifPresent == null) {
                ifPresent = new SumAndCount();
                _countMap.put(counter, ifPresent);
            }

            ifPresent.addDelta(delta);

            if (ifPresent._count >= _queueForTheseManyCalls) {
                CounterDTO dto = CounterDTO.builder()
                        .name(counter)
                        .delta(ifPresent._sum)
                        .build();
                ifPresent.clear();
                _httpMetrics.count(dto);
            }
        }
    }

    @Override
    public void gauge(String gauge, double value) {

        if (gauge == null || gauge.trim().isEmpty()) {
            return;
        }

        synchronized (_gaugeLock) {
            ValueAndCount ifPresent = _gaugeMap.get(gauge);
            if (ifPresent == null) {
                ifPresent = new ValueAndCount();
                _gaugeMap.put(gauge, ifPresent);
            }

            ifPresent.setValue(value);

            if (ifPresent._count >= _queueForTheseManyCalls) {
                GaugeDTO dto = GaugeDTO.builder()
                        .name(gauge)
                        .value(ifPresent._value)
                        .build();
                ifPresent.clear();
                _httpMetrics.gauge(dto);
            }
        }
    }

    @Override
    public void time(String operation, long timeInMs) {
        if (operation == null || operation.trim().isEmpty() || timeInMs < 0L) {
            // error
            return;
        }
        synchronized (_latencyLock) {
            if (!_latencyMap.containsKey(operation)) {
                List<Long> list = Lists.newArrayListWithExpectedSize(_queueForTheseManyCalls);
                _latencyMap.put(operation, list);
            }

            List<Long> list = _latencyMap.get(operation);
            list.add(timeInMs);

            if (list.size() >= _queueForTheseManyCalls) {

                LatencyDTO dto = LatencyDTO.builder()
                        .name(operation)
                        .latencies(Lists.newArrayList(list))
                        .build();

                list.clear();

                _httpMetrics.time(dto);

            }
        }
    }


    private static final class SumAndCount {
        private int _count = 0;
        private long _sum = 0L;

        public void addDelta(long delta) {
            _count++;
            _sum += delta;
        }

        public void clear() {
            _count = 0;
            _sum = 0L;
        }

    }

    private static final class ValueAndCount {
        private int _count = 0;
        private double _value= 0L;

        public void setValue(double value) {
            _count++;
            _value = value;
        }

        public void clear() {
            _count = 0;
            _value = 0d;
        }

    }

}
