/*
 * Decompiled with CFR 0.152.
 */
package no.digipost.cache2.fallback.disk;

import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import no.digipost.cache2.function.ThrowingRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LockedFile {
    static final Duration DEFAULT_EXPIRY_TIME = Duration.ofMinutes(10L);
    static final Logger LOG = LoggerFactory.getLogger(LockedFile.class);
    private static final String LOCK_FILE_POSTFIX = "." + LockedFile.class.getCanonicalName() + ".lock";
    private final Path file;
    private final Path lockfile;
    private final Duration maximumLockingDuration;
    private final Clock clock;

    LockedFile(Path forFile, Clock clock) {
        this(forFile, DEFAULT_EXPIRY_TIME, clock);
    }

    LockedFile(Path forFile, Duration maximumLockingDuration, Clock clock) {
        this.file = forFile;
        this.lockfile = forFile.resolveSibling(forFile.getFileName() + LOCK_FILE_POSTFIX);
        this.maximumLockingDuration = maximumLockingDuration;
        this.clock = clock;
    }

    public Path getPath() {
        return this.file;
    }

    public <X extends Exception> boolean runIfLock(ThrowingRunnable<X> operation) throws X {
        if (this.tryLock()) {
            try {
                operation.run();
                boolean bl = true;
                return bl;
            }
            finally {
                this.release();
            }
        }
        return false;
    }

    public boolean tryLock() {
        if (this.isLocked()) {
            if (this.isExpiredAt(this.clock.instant())) {
                LOG.warn("Lock-file is considered to be expired since it is older than {}. Deleting it. There may be some problematic code which are unable to correctly release an aquired lock.", (Object)this.maximumLockingDuration);
                this.releaseExpired();
            } else {
                LOG.debug("Another process is updating the cache-value. Not yielding lock.");
                return false;
            }
        }
        try {
            LOG.trace("Creating lockfile");
            Files.createFile(this.lockfile, new FileAttribute[0]);
            LOG.trace("Acquired lock.");
            return true;
        }
        catch (FileAlreadyExistsException fileAlreadyExists) {
            LOG.debug("Failed to create lock-file because {}: '{}'. Means that another process created it. Will not run delegate.", (Object)fileAlreadyExists.getClass().getSimpleName(), (Object)fileAlreadyExists.getMessage());
            return false;
        }
        catch (IOException e) {
            throw new UnableToAcquireLock(e, this.maximumLockingDuration);
        }
    }

    public boolean isLocked() {
        return Files.exists(this.lockfile, new LinkOption[0]);
    }

    private boolean isExpiredAt(Instant instant) {
        try {
            Instant lastModifiedTime = Files.getLastModifiedTime(this.lockfile, new LinkOption[0]).toInstant();
            return lastModifiedTime.isBefore(instant.minus(this.maximumLockingDuration));
        }
        catch (IOException e) {
            LOG.warn("Failed to read last-modified time of lock-file because {}: '{}'. Treats is as not expired.", (Object)e.getClass().getSimpleName(), (Object)e.getMessage());
            return false;
        }
    }

    private boolean releaseExpired() {
        try {
            LOG.trace("Deleting expired lockfile");
            if (!Files.deleteIfExists(this.lockfile)) {
                LOG.info("Another process may have deleted the expired lock-file. This is expected behavior, and continuing normally.");
            }
            return true;
        }
        catch (IOException e) {
            throw new UnableToReleaseLock(e, this.maximumLockingDuration);
        }
    }

    public void release() {
        try {
            LOG.trace("Deleting lockfile");
            Files.delete(this.lockfile);
        }
        catch (NoSuchFileException e) {
            throw new TryingToDeleteNonExistingLockFile(e);
        }
        catch (IOException e) {
            throw new UnableToReleaseLock(e, this.maximumLockingDuration);
        }
    }

    public static class UnableToReleaseLock
    extends RuntimeException {
        private UnableToReleaseLock(Exception cause, Duration lockExpiryDuration) {
            super("Unable to delete lock-file because " + cause.getClass().getSimpleName() + ": '" + cause.getMessage() + "'. The lock may now not be acquired until it expires (expiry time: " + lockExpiryDuration + ").", cause);
        }
    }

    public static class TryingToDeleteNonExistingLockFile
    extends RuntimeException {
        public TryingToDeleteNonExistingLockFile(NoSuchFileException cause) {
            super("Got " + cause.getClass().getSimpleName() + ": '" + cause.getMessage() + "' when trying to delete lock-file. This could indicate that the lock-file was deleted by another process, and indicates a bug. Tt should never happen as long as other processes honor the lock-file.", cause);
        }
    }

    public static class UnableToAcquireLock
    extends RuntimeException {
        private UnableToAcquireLock(Exception cause, Duration lockExpiryDuration) {
            super("Got " + cause.getClass().getSimpleName() + ": '" + cause.getMessage() + "' when trying to create lock-file, and thus the lock will not be yielded. In the unlikely event that the lock-file was created after all, it will prevent anyone from acquiring the lock until it has expired in " + lockExpiryDuration, cause);
        }
    }
}

