/*
 * Decompiled with CFR 0.152.
 */
package no.digipost.sanitizing.internal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import no.digipost.sanitizing.exception.CSSValidationException;
import no.digipost.sanitizing.internal.ApiHtmlValidatorPolicy;
import org.owasp.html.HtmlStreamEventProcessor;
import org.owasp.html.HtmlStreamEventReceiver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class StyleElementPreprocessor
implements HtmlStreamEventProcessor {
    StyleElementPreprocessor() {
    }

    public HtmlStreamEventReceiver wrap(HtmlStreamEventReceiver receiver) {
        return new StyleElementReceiver(receiver);
    }

    public static class StyleElementReceiver
    implements HtmlStreamEventReceiver {
        private static final String STYLE_TAG = "style";
        private static final Set<String> WORDS_IN_VALUE_BLACKLIST = new HashSet<String>(Arrays.asList("javascript", "expression", "url(", "http://", "https://", "/*", "*/"));
        private static final Map<String, String> HTML_ESCAPE_CHARS = new HashMap<String, String>(){
            {
                this.put("&", "&amp;");
                this.put("<", "&lt;");
                this.put(">", "&gt;");
                this.put("/", "&#x2F;");
            }
        };
        private static final Pattern completeCssPattern = Pattern.compile("(?:([\\.\\#\\-\\w\\s\\: \\[\\],]+)\\s*\\{([^}]+)\\}\\s*)+");
        private static final Pattern selectorAndContentPattern = Pattern.compile("([\\.\\#\\-\\w\\s\\: \\[\\],]+)\\s*\\{([^}]+)\\}\\s*");
        private static final Pattern propertyValuePattern = Pattern.compile("\\s*([\\w- ]*)\\:([\\w\\d\\. \\,\\%\\#\\-\\:\\\"\\'\\(\\)\\!\\\\]*)\\;");
        private static final Logger log = LoggerFactory.getLogger(StyleElementReceiver.class);
        private final HtmlStreamEventReceiver receiver;
        private boolean inStyleTag;

        StyleElementReceiver(HtmlStreamEventReceiver receiver) {
            this.receiver = receiver;
        }

        public static String validateAndSanitizeCss(String css) {
            ArrayList<String> validationErrors = new ArrayList<String>();
            StyleElementReceiver.validateCss(css, validationErrors);
            if (!validationErrors.isEmpty()) {
                throw new CSSValidationException(validationErrors);
            }
            return StyleElementReceiver.sanitizeCharsToHtmlEscapedChars(css);
        }

        private static void validateCss(String css, List<String> validationErrors) {
            String cssStriped = css.trim();
            Matcher validCss = completeCssPattern.matcher(cssStriped);
            if (!validCss.matches() && cssStriped.length() > 0) {
                validationErrors.add("CSS in style-element is invalid.");
            }
            Matcher selectorAndContent = selectorAndContentPattern.matcher(cssStriped);
            while (selectorAndContent.find()) {
                String selector = selectorAndContent.group(1).trim();
                String selectorContent = selectorAndContent.group(2).trim();
                log.debug("Parsing selectorContent for selector {}", (Object)selector);
                if (selector.length() == 0) {
                    log.warn("Could not match css-selector, but regex matched anyways. Something might be wrong with the regex. Style-element content: {}", (Object)cssStriped);
                    validationErrors.add("CSS selector not found. Indicates illegal css.");
                    continue;
                }
                if (StyleElementReceiver.containsBlacklistedWord(selectorContent)) {
                    validationErrors.add("Content of selector '" + selector + "' contains one or more illegal words.");
                    continue;
                }
                StyleElementReceiver.validateDeclarations(selectorContent, validationErrors);
            }
        }

        private static boolean containsBlacklistedWord(String content) {
            String contentLowerCase = content.toLowerCase();
            return WORDS_IN_VALUE_BLACKLIST.stream().anyMatch(blacklistedWord -> contentLowerCase.contains(blacklistedWord.toLowerCase()));
        }

        private static void validateDeclarations(String selectorContent, List<String> validationErrors) {
            String stripedContent = selectorContent.trim();
            Matcher declaration = propertyValuePattern.matcher(stripedContent);
            boolean matcherDidNotFindDeclaration = true;
            while (declaration.find()) {
                matcherDidNotFindDeclaration = false;
                String property = declaration.group(1).trim();
                String value = declaration.group(2).trim();
                log.debug("Processing declaration:  {}: {}", (Object)property, (Object)value);
                if (ApiHtmlValidatorPolicy.CSS_PROPERTY_WHITELIST.containsKey(property)) {
                    boolean isNotOK = !ApiHtmlValidatorPolicy.CSS_PROPERTY_WHITELIST.get(property).test(value.toLowerCase());
                    if (!isNotOK) continue;
                    log.warn("Value '{}' is not allowed for property '{}'.", (Object)value, (Object)property);
                    validationErrors.add("Value '" + value + "' is not allowed for property '" + property + "'.");
                    continue;
                }
                log.debug("Property '{}' is not allowed in style-element", (Object)property);
                validationErrors.add("Property '" + property + "' is not allowed.");
            }
            if (matcherDidNotFindDeclaration && stripedContent.length() > 0) {
                if (!stripedContent.endsWith(";")) {
                    validationErrors.add("Declaration must end with ';'.");
                } else {
                    log.warn("Found no match in property and value regex, even though it should. Something might be wrong with the regex. Content: {}", (Object)stripedContent);
                    validationErrors.add("Malformed declaration: " + stripedContent + " .");
                }
            }
        }

        private static String sanitizeCharsToHtmlEscapedChars(String css) {
            String escapedStyleContent = css;
            for (Map.Entry<String, String> charAndEscaped : HTML_ESCAPE_CHARS.entrySet()) {
                String character = charAndEscaped.getKey();
                String escapedChar = charAndEscaped.getValue();
                escapedStyleContent = escapedStyleContent.replace(character, escapedChar);
            }
            return escapedStyleContent;
        }

        public void openDocument() {
            this.receiver.openDocument();
            this.inStyleTag = false;
        }

        public void closeDocument() {
            this.receiver.closeDocument();
        }

        public void openTag(String elementName, List<String> attrs) {
            this.receiver.openTag(elementName, attrs);
            this.inStyleTag = STYLE_TAG.equals(elementName);
        }

        public void closeTag(String elementName) {
            this.receiver.closeTag(elementName);
            this.inStyleTag = false;
        }

        public void text(String text) {
            if (this.inStyleTag) {
                this.receiver.text(StyleElementReceiver.validateAndSanitizeCss(text));
            } else {
                this.receiver.text(text);
            }
        }
    }
}

