/*
 * Decompiled with CFR 0.152.
 */
package no.digipost.print.validate;

import java.awt.geom.Rectangle2D;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import no.digipost.print.validate.PDFBoxConfigurer;
import no.digipost.print.validate.PdfFontValidator;
import no.digipost.print.validate.PdfValidationError;
import no.digipost.print.validate.PdfValidationResult;
import no.digipost.print.validate.PdfValidationSettings;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.encryption.InvalidPasswordException;
import org.apache.pdfbox.pdmodel.font.PDFont;
import org.apache.pdfbox.text.PDFTextStripperByArea;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfValidator {
    private static final Logger LOG = LoggerFactory.getLogger(PdfValidator.class);
    private final PdfFontValidator fontValidator = new PdfFontValidator();
    private static final double MM_TO_POINTS = 2.8346457481384277;
    public static final int A4_HEIGHT_MM = 297;
    public static final int A4_WIDTH_MM = 210;
    public static final int BARCODE_AREA_WIDTH_MM = 15;
    public static final int BARCODE_AREA_HEIGHT_MM = 80;
    public static final int BARCODE_AREA_X_POS_MM = 0;
    public static final int BARCODE_AREA_Y_POS_MM = 95;
    public static final List<Float> PDF_VERSIONS_SUPPORTED_FOR_PRINT;

    public PdfValidationResult validate(byte[] pdfContent, PdfValidationSettings printValidationSettings) {
        return this.validateForPrint(new ByteArrayInputStream(pdfContent), printValidationSettings);
    }

    public PdfValidationResult validate(Path pdfFile, PdfValidationSettings printValidationSettings) throws IOException {
        try (InputStream pdfStream = this.openFileAsInputStream(pdfFile);){
            PdfValidationResult pdfValidationResult = this.validateForPrint(pdfStream, printValidationSettings);
            return pdfValidationResult;
        }
    }

    private PdfValidationResult validateForPrint(InputStream pdfStream, PdfValidationSettings printValidationSettings) {
        List<PdfValidationError> errors;
        int numberOfPages = -1;
        try (PDDocument pdDoc = PDDocument.load((InputStream)pdfStream);){
            numberOfPages = pdDoc.getNumberOfPages();
            errors = this.validateDocumentForPrint(pdDoc, printValidationSettings);
        }
        catch (InvalidPasswordException invalidPassword) {
            errors = this.failValidationIfEncrypted(new ArrayList<PdfValidationError>());
        }
        catch (Exception e) {
            errors = Arrays.asList(PdfValidationError.PDF_PARSE_ERROR);
            LOG.debug("PDF could not be parsed. ({}: '{}')", new Object[]{e.getClass().getSimpleName(), e.getMessage(), e});
        }
        return new PdfValidationResult(errors, numberOfPages, printValidationSettings.bleed);
    }

    List<PdfValidationError> validateDocumentForPrint(PDDocument pdDoc, PdfValidationSettings settings) throws IOException {
        ArrayList<PdfValidationError> errors = new ArrayList<PdfValidationError>();
        if (pdDoc.isEncrypted()) {
            return this.failValidationIfEncrypted(errors);
        }
        if (settings.validateNumberOfPages) {
            this.validerSideantall(pdDoc.getNumberOfPages(), settings.maxNumberOfPages, errors);
        }
        if (settings.validatePDFversion) {
            this.validatePdfVersion(pdDoc.getDocument().getVersion(), errors);
        }
        boolean documentHasInvalidDimensions = false;
        for (PDPage page : pdDoc.getPages()) {
            if (!this.hasInvalidDimensions(page, settings.bleed)) continue;
            documentHasInvalidDimensions = true;
            break;
        }
        this.addValidationError(documentHasInvalidDimensions, PdfValidationError.UNSUPPORTED_DIMENSIONS, errors);
        boolean hasTextInBarcodeArea = false;
        boolean documentContainsPagesWithInvalidPrintMargins = false;
        if (settings.validateLeftMargin) {
            for (PDPage page : pdDoc.getPages()) {
                try {
                    if (!this.hasTextInBarcodeArea(page, settings.bleed)) continue;
                    hasTextInBarcodeArea = true;
                    break;
                }
                catch (Exception npe) {
                    documentContainsPagesWithInvalidPrintMargins = true;
                    LOG.debug("Unable to validate the margin on one of the pages.", (Throwable)npe);
                }
            }
        }
        this.addValidationError(documentContainsPagesWithInvalidPrintMargins, PdfValidationError.UNABLE_TO_VERIFY_SUITABLE_MARGIN_FOR_PRINT, errors);
        this.addValidationError(hasTextInBarcodeArea, PdfValidationError.INSUFFICIENT_MARGIN_FOR_PRINT, errors);
        if (settings.validateFonts) {
            for (PDPage page : pdDoc.getPages()) {
                this.validateFonts(this.fontValidator.getPageFonts(page), errors);
            }
        }
        return errors;
    }

    private void addValidationError(boolean documentContainsPagesThatCannotBeParsed, PdfValidationError validationErrors, List<PdfValidationError> errors) {
        if (documentContainsPagesThatCannotBeParsed) {
            errors.add(validationErrors);
        }
    }

    private List<PdfValidationError> failValidationIfEncrypted(List<PdfValidationError> errors) {
        errors.add(PdfValidationError.PDF_IS_ENCRYPTED);
        LOG.debug("The pdf is encrypted.");
        return errors;
    }

    private void validateFonts(Iterable<PDFont> fonter, List<PdfValidationError> errors) {
        List<PDFont> nonSupportedFonts = this.fontValidator.findNonSupportedFonts(fonter);
        if (!nonSupportedFonts.isEmpty()) {
            errors.add(PdfValidationError.REFERENCES_INVALID_FONT);
            if (LOG.isInfoEnabled()) {
                LOG.debug("The PDF has references to invalid fonts: [{}]", (Object)nonSupportedFonts.stream().map(this::describe).collect(Collectors.joining(", ")));
            }
        }
    }

    private String describe(PDFont font) {
        return font.getSubType() + " '" + font.getName() + "'";
    }

    private void validatePdfVersion(float pdfVersion, List<PdfValidationError> errors) {
        if (!PDF_VERSIONS_SUPPORTED_FOR_PRINT.contains(Float.valueOf(pdfVersion))) {
            errors.add(PdfValidationError.UNSUPPORTED_PDF_VERSION_FOR_PRINT);
            LOG.info("PDF version was {}. {}", (Object)Float.valueOf(pdfVersion), (Object)PdfValidationError.UNSUPPORTED_PDF_VERSION_FOR_PRINT);
        }
    }

    private void validerSideantall(int numberOfPages, int maxPages, List<PdfValidationError> errors) {
        if (numberOfPages > maxPages) {
            errors.add(PdfValidationError.TOO_MANY_PAGES_FOR_AUTOMATED_PRINT);
            LOG.debug("The PDF has too many pages. Max number of pages is {}. Actual number of pages is {}", (Object)maxPages, (Object)numberOfPages);
        }
        if (numberOfPages == 0) {
            errors.add(PdfValidationError.DOCUMENT_HAS_NO_PAGES);
            LOG.debug("The PDF document does not contain any pages. The file may be corrupt.", (Object)numberOfPages);
        }
    }

    private boolean hasTextInBarcodeArea(PDPage pdPage, PdfValidationSettings.Bleed bleed) throws IOException {
        SilentZone silentZone = new SilentZone(pdPage.getCropBox(), bleed);
        Rectangle2D.Double leftMarginBarcodeArea = new Rectangle2D.Double(silentZone.upperLeftCornerX, silentZone.upperLeftCornerY, silentZone.silentZoneXSize, silentZone.silentZoneYSize);
        return this.hasTextInArea(pdPage, leftMarginBarcodeArea);
    }

    private boolean hasInvalidDimensions(PDPage page, PdfValidationSettings.Bleed bleed) {
        PDRectangle findCropBox = page.getCropBox();
        long pageHeightInMillimeters = PdfValidator.pointsTomm(findCropBox.getHeight());
        long pageWidthInMillimeters = PdfValidator.pointsTomm(findCropBox.getWidth());
        if (!PdfValidator.isPortraitA4(pageWidthInMillimeters, pageHeightInMillimeters, bleed) && !PdfValidator.isLandscapeA4(pageWidthInMillimeters, pageHeightInMillimeters, bleed)) {
            LOG.debug("One or more pages in the PDF has invalid dimensions.  Valid dimensions are width {} mm and height {} mm, alt width {} mm og height {} mm with {} mm lower flexibility and {} upper flexibility. Actual dimensions are width: {} mm and height: {} mm.", new Object[]{210, 297, 297, 210, bleed.negativeBleedInMM, bleed.positiveBleedInMM, pageWidthInMillimeters, pageHeightInMillimeters});
            return true;
        }
        return false;
    }

    private static boolean isPortraitA4(long pageWidthInMillimeters, long pageHeightInMillimeters, PdfValidationSettings.Bleed bleed) {
        long minimumWidth = 210 - bleed.negativeBleedInMM;
        long maximumWidth = 210 + bleed.positiveBleedInMM;
        long minimumHeight = 297 - bleed.negativeBleedInMM;
        long maximumHeight = 297 + bleed.positiveBleedInMM;
        return pageWidthInMillimeters <= maximumWidth && pageWidthInMillimeters >= minimumWidth && pageHeightInMillimeters <= maximumHeight && pageHeightInMillimeters >= minimumHeight;
    }

    private static boolean isLandscapeA4(long pageWidthInMillimeters, long pageHeightInMillimeters, PdfValidationSettings.Bleed bleed) {
        return PdfValidator.isPortraitA4(pageHeightInMillimeters, pageWidthInMillimeters, bleed);
    }

    private boolean hasTextInArea(PDPage pdPage, Rectangle2D area) throws IOException {
        boolean hasTextInArea = false;
        PDFTextStripperByArea stripper = new PDFTextStripperByArea();
        stripper.addRegion("marginArea", area);
        stripper.extractRegions(pdPage);
        String text = stripper.getTextForRegion("marginArea");
        if (text != null && text.trim().length() > 0) {
            hasTextInArea = true;
        }
        return hasTextInArea;
    }

    private InputStream openFileAsInputStream(Path pdfFile) throws IOException {
        return new BufferedInputStream(Files.newInputStream(pdfFile, new OpenOption[0]));
    }

    private static double mmToPoints(int sizeInMillimeters) {
        BigDecimal points = new BigDecimal((double)sizeInMillimeters * 2.8346457481384277);
        points = points.setScale(1, RoundingMode.DOWN);
        return points.doubleValue();
    }

    private static long pointsTomm(double sizeInPoints) {
        return Math.round(sizeInPoints / 2.8346457481384277);
    }

    static {
        PDFBoxConfigurer.configure();
        PDF_VERSIONS_SUPPORTED_FOR_PRINT = Arrays.asList(Float.valueOf(1.0f), Float.valueOf(1.1f), Float.valueOf(1.2f), Float.valueOf(1.3f), Float.valueOf(1.4f), Float.valueOf(1.5f), Float.valueOf(1.6f), Float.valueOf(1.7f));
    }

    private static class SilentZone {
        public final double upperLeftCornerX;
        public final double upperLeftCornerY;
        public final double silentZoneXSize;
        public final double silentZoneYSize;

        private SilentZone(PDRectangle findCropBox, PdfValidationSettings.Bleed bleed) {
            int pageHeightInMillimeters = (int)PdfValidator.pointsTomm(findCropBox.getHeight());
            int pageWidthInMillimeters = (int)PdfValidator.pointsTomm(findCropBox.getWidth());
            boolean isLandscape = PdfValidator.isLandscapeA4(pageWidthInMillimeters, pageHeightInMillimeters, bleed);
            this.upperLeftCornerX = this.upperLeftCornerX(isLandscape);
            this.upperLeftCornerY = this.upperLeftCornerY(isLandscape, pageHeightInMillimeters);
            this.silentZoneXSize = this.zoneXSize(isLandscape);
            this.silentZoneYSize = this.zoneYSize(isLandscape);
        }

        private double upperLeftCornerX(boolean isLandscape) {
            if (isLandscape) {
                return PdfValidator.mmToPoints(95);
            }
            return PdfValidator.mmToPoints(0);
        }

        private double upperLeftCornerY(boolean isLandscape, int pageHeightInMillimeters) {
            if (isLandscape) {
                return PdfValidator.mmToPoints(pageHeightInMillimeters - 15);
            }
            return PdfValidator.mmToPoints(95);
        }

        private double zoneXSize(boolean isLandscape) {
            if (isLandscape) {
                return PdfValidator.mmToPoints(80);
            }
            return PdfValidator.mmToPoints(15);
        }

        private double zoneYSize(boolean isLandscape) {
            if (isLandscape) {
                return PdfValidator.mmToPoints(15);
            }
            return PdfValidator.mmToPoints(80);
        }
    }
}

