/*
 * SonarQube Java
 * Copyright (C) 2012 SonarSource
 * dev@sonar.codehaus.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.checks;

import com.google.common.annotations.Beta;
import org.sonar.api.rule.RuleKey;
import org.sonar.check.BelongsToProfile;
import org.sonar.check.Priority;
import org.sonar.check.Rule;
import org.sonar.plugins.java.api.JavaFileScanner;
import org.sonar.plugins.java.api.JavaFileScannerContext;
import org.sonar.plugins.java.api.tree.BaseTreeVisitor;
import org.sonar.plugins.java.api.tree.BlockTree;
import org.sonar.plugins.java.api.tree.LambdaExpressionTree;
import org.sonar.plugins.java.api.tree.Tree;

@Rule(
    key = LambdaSingleExpressionCheck.RULE_KEY,
    priority = Priority.MAJOR,
    tags={"java8"})
@BelongsToProfile(title = "Sonar way", priority = Priority.MAJOR)
@Beta
public class LambdaSingleExpressionCheck extends BaseTreeVisitor implements JavaFileScanner {

  public static final String RULE_KEY = "S1602";
  private static final RuleKey RULE = RuleKey.of(CheckList.REPOSITORY_KEY, RULE_KEY);

  private JavaFileScannerContext context;

  @Override
  public void scanFile(JavaFileScannerContext context) {
    this.context = context;
    scan(context.getTree());
  }

  @Override
  public void visitLambdaExpression(LambdaExpressionTree lambdaExpressionTree) {
    if (isBlockWithOneStatement(lambdaExpressionTree)) {
      String message = "Remove useless curly braces around statement";
      if(singleStatementIsReturn(lambdaExpressionTree)){
        message += " and then remove useless return keyword";
      }
      context.addIssue(lambdaExpressionTree.body(), RULE, message);
    }
    super.visitLambdaExpression(lambdaExpressionTree);
  }

  private boolean isBlockWithOneStatement(LambdaExpressionTree lambdaExpressionTree) {
    return lambdaExpressionTree.body().is(Tree.Kind.BLOCK) && ((BlockTree) lambdaExpressionTree.body()).body().size() == 1;
  }

  private boolean singleStatementIsReturn(LambdaExpressionTree lambdaExpressionTree) {
    return ((BlockTree) lambdaExpressionTree.body()).body().get(0).is(Tree.Kind.RETURN_STATEMENT);
  }


}
