/*
 * SonarQube Java
 * Copyright (C) 2012 SonarSource
 * dev@sonar.codehaus.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.checks;

import com.google.common.collect.ImmutableList;
import org.sonar.check.Priority;
import org.sonar.check.Rule;
import org.sonar.plugins.java.api.tree.ClassTree;
import org.sonar.plugins.java.api.tree.Modifier;
import org.sonar.plugins.java.api.tree.Tree;
import org.sonar.plugins.java.api.tree.VariableTree;

import java.util.List;

@Rule(
    key = "S2039",
    priority = Priority.MAJOR,
    tags = {"security"})
public class FieldModifierCheck extends SubscriptionBaseVisitor {

  @Override
  public List<Tree.Kind> nodesToVisit() {
    return ImmutableList.of(Tree.Kind.CLASS, Tree.Kind.ENUM);
  }

  @Override
  public void visitNode(Tree tree) {
    ClassTree classTree = (ClassTree) tree;
    for (Tree member : classTree.members()) {
      if(member.is(Tree.Kind.VARIABLE) && hasNoVisibilityModifier((VariableTree) member)) {
        addIssue(member, "Explicitly declare the visibility for \""+((VariableTree) member).simpleName().name()+"\".");
      }
    }
  }

  private boolean hasNoVisibilityModifier(VariableTree member) {
    return !(hasModifier(member, Modifier.PUBLIC) || hasModifier(member, Modifier.PRIVATE) || hasModifier(member, Modifier.PROTECTED));
  }

  private boolean hasModifier(VariableTree member, Modifier modifier) {
    return member.modifiers().modifiers().contains(modifier);
  }
}
