/*
 * Decompiled with CFR 0.152.
 */
package org.expath.pkg.repo;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;
import org.expath.pkg.repo.Extension;
import org.expath.pkg.repo.FileHelper;
import org.expath.pkg.repo.FileSystemStorage;
import org.expath.pkg.repo.Package;
import org.expath.pkg.repo.PackageException;
import org.expath.pkg.repo.Packages;
import org.expath.pkg.repo.Storage;
import org.expath.pkg.repo.URISpace;
import org.expath.pkg.repo.Universe;
import org.expath.pkg.repo.UserInteractionStrategy;
import org.expath.pkg.repo.XarSource;
import org.expath.pkg.repo.XarUriSource;
import org.expath.pkg.repo.ZipHelper;
import org.expath.pkg.repo.parser.DescriptorParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class Repository
implements Universe {
    private final Storage myStorage;
    @GuardedBy(value="myPackages")
    private final Map<String, Packages> myPackages = new HashMap<String, Packages>();
    private final ReadWriteLock myPackagesLock = new ReentrantReadWriteLock();
    @GuardedBy(value="myExtensionsLock")
    private final Map<String, Extension> myExtensions = new HashMap<String, Extension>();
    private final ReadWriteLock myExtensionsLock = new ReentrantReadWriteLock();
    private static final Logger LOG = LoggerFactory.getLogger(Repository.class);

    public Repository(Storage storage) {
        LOG.info("Create a new repository with storage: {}", (Object)storage);
        this.myStorage = storage;
    }

    public List<PackageException> init() {
        List<PackageException> exceptions = this.loadExtensions();
        List<PackageException> parseExceptions = this.parsePublicUris();
        if (parseExceptions != Collections.emptyList()) {
            if (exceptions == Collections.emptyList()) {
                return parseExceptions;
            }
            exceptions.addAll(parseExceptions);
        }
        return exceptions;
    }

    private List<PackageException> loadExtensions() {
        ArrayList<PackageException> exceptions = null;
        ServiceLoader<Extension> loader = ServiceLoader.load(Extension.class);
        for (Extension extension : loader) {
            try {
                this.registerExtension(extension);
            }
            catch (PackageException e) {
                if (exceptions == null) {
                    exceptions = new ArrayList<PackageException>();
                }
                exceptions.add(e);
            }
        }
        if (exceptions != null) {
            return exceptions;
        }
        return Collections.emptyList();
    }

    public static Repository makeDefaultRepo() throws PackageException {
        return Repository.makeDefaultRepo(null);
    }

    public static Repository makeDefaultRepo(String dir) throws PackageException {
        if (dir == null) {
            dir = System.getProperty("expath.repo");
        }
        if (dir == null) {
            dir = System.getenv("EXPATH_REPO");
        }
        if (dir != null) {
            Path f = Paths.get(dir, new String[0]);
            if (!Files.exists(f, new LinkOption[0])) {
                throw new PackageException("Repo directory does not exist: " + dir);
            }
            if (!Files.isDirectory(f, new LinkOption[0])) {
                throw new PackageException("Repo is not a directory: " + dir);
            }
            try {
                FileSystemStorage storage = new FileSystemStorage(f);
                return new Repository(storage);
            }
            catch (PackageException ex) {
                throw new PackageException("Error setting the repo (" + dir + ")", ex);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void registerExtension(Extension ext) throws PackageException {
        boolean registeredExtension = false;
        this.myExtensionsLock.writeLock().lock();
        try {
            if (!this.myExtensions.containsKey(ext.getName())) {
                this.myExtensions.put(ext.getName(), ext);
                registeredExtension = true;
            }
        }
        finally {
            this.myExtensionsLock.writeLock().unlock();
        }
        if (registeredExtension) {
            this.myPackagesLock.readLock().lock();
            try {
                ext.init(this, this.myPackages);
            }
            finally {
                this.myPackagesLock.readLock().unlock();
            }
        }
    }

    public List<PackageException> reload() {
        this.myPackagesLock.writeLock().lock();
        try {
            this.myPackages.clear();
        }
        finally {
            this.myPackagesLock.writeLock().unlock();
        }
        return this.parsePublicUris();
    }

    public Storage getStorage() {
        return this.myStorage;
    }

    public Collection<Packages> listPackages() {
        this.myPackagesLock.readLock().lock();
        try {
            Collection<Packages> collection = this.myPackages.values();
            return collection;
        }
        finally {
            this.myPackagesLock.readLock().unlock();
        }
    }

    public Packages getPackages(String name) {
        this.myPackagesLock.readLock().lock();
        try {
            Packages packages = this.myPackages.get(name);
            return packages;
        }
        finally {
            this.myPackagesLock.readLock().unlock();
        }
    }

    public static Repository createRepository(Path dir) throws PackageException {
        if (Files.exists(dir, new LinkOption[0])) {
            if (!Files.isDirectory(dir, new LinkOption[0])) {
                throw new PackageException("File exists and is not a directory (" + dir + ")");
            }
            if (!FileHelper.isEmpty(dir)) {
                throw new PackageException("Directory exists and is not empty (" + dir + ")");
            }
        } else {
            try {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new PackageException("Error creating the directory (" + dir + ")", e);
            }
        }
        Path priv_dir = dir.resolve(".expath-pkg");
        try {
            Files.createDirectories(priv_dir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new PackageException("Error creating the private directory (" + priv_dir + ")", e);
        }
        return new Repository(new FileSystemStorage(dir));
    }

    public Package installPackage(URI pkg, boolean force, UserInteractionStrategy interact) throws PackageException {
        if (this.myStorage.isReadOnly()) {
            throw new PackageException("The storage is read-only, package install not supported");
        }
        return this.installPackage(new XarUriSource(pkg), force, interact);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Package installPackage(XarSource xarSource, boolean force, UserInteractionStrategy interact) throws PackageException {
        Packages pp;
        String version;
        String name;
        Package pkg;
        Path tmp_dir;
        block34: {
            if (!xarSource.isValid()) {
                throw new PackageException("Package file does not exist (" + xarSource.getURI() + ")");
            }
            this.myStorage.beforeInstall(force, interact);
            tmp_dir = this.myStorage.makeTempDir("install");
            try {
                ZipHelper zip = new ZipHelper(xarSource);
                zip.unzip(tmp_dir);
            }
            catch (IOException ex) {
                throw new PackageException("Error unziping the package", ex);
            }
            interact.logInfo("Package unziped to " + tmp_dir);
            Path desc_f = tmp_dir.resolve("expath-pkg.xml");
            if (!Files.exists(desc_f, new LinkOption[0])) {
                throw new PackageException("Package descriptor does NOT exist in: " + tmp_dir);
            }
            try (InputStream is = Files.newInputStream(desc_f, new OpenOption[0]);){
                StreamSource desc = new StreamSource(is);
                DescriptorParser parser = new DescriptorParser();
                pkg = parser.parse(desc, null, this.myStorage, this);
            }
            catch (IOException e) {
                throw new PackageException(e.getMessage(), e);
            }
            name = pkg.getName();
            version = pkg.getVersion();
            pp = null;
            this.myPackagesLock.writeLock().lock();
            try {
                Package p2;
                pp = this.myPackages.get(name);
                if (pp == null || (p2 = pp.version(version)) == null) break block34;
                if (force || interact.ask("Force override " + name + " - " + version + "?", false)) {
                    this.myStorage.remove(p2);
                    pp.remove(p2);
                    if (pp.latest() == null) {
                        this.myPackages.remove(name);
                    }
                    break block34;
                }
                throw new AlreadyInstalledException(name, version);
            }
            finally {
                this.myPackagesLock.writeLock().unlock();
            }
        }
        String key = pkg.getAbbrev() + "-" + version;
        for (int i = 1; this.myStorage.packageKeyExists(key) && i < 100; ++i) {
            key = pkg.getAbbrev() + "-" + version + "__" + i;
        }
        if (this.myStorage.packageKeyExists(key)) {
            String msg = "Impossible to find a non-existing package key in the repo, stopped at: ";
            throw new PackageException(msg + key);
        }
        this.myStorage.storeInstallDir(tmp_dir, key, pkg);
        if (pp == null) {
            pp = new Packages(name);
            this.myPackagesLock.writeLock().lock();
            try {
                this.myPackages.put(name, pp);
            }
            finally {
                this.myPackagesLock.writeLock().unlock();
            }
        }
        pp.add(pkg);
        this.myStorage.updatePackageLists(pkg);
        this.myExtensionsLock.readLock().lock();
        try {
            for (Extension ext : this.myExtensions.values()) {
                ext.install(this, pkg);
            }
        }
        finally {
            this.myExtensionsLock.readLock().unlock();
        }
        return pkg;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removePackage(String pkg, boolean force, UserInteractionStrategy interact) throws PackageException {
        if (!interact.ask("Remove package " + pkg + "?", true)) {
            return false;
        }
        this.myPackagesLock.writeLock().lock();
        try {
            Packages pp = this.myPackages.get(pkg);
            if (pp == null) {
                if (force) {
                    boolean bl = false;
                    return bl;
                }
                throw new PackageException("The package does not exist: " + pkg);
            }
            if (pp.packages().size() != 1) {
                throw new PackageException("The package has several versions installed: " + pkg);
            }
            Package p = pp.latest();
            this.myStorage.remove(p);
            pp.remove(p);
            this.myPackages.remove(pkg);
        }
        finally {
            this.myPackagesLock.writeLock().unlock();
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removePackage(String pkg, String version, boolean force, UserInteractionStrategy interact) throws PackageException {
        if (!interact.ask("Remove package " + pkg + ", version " + version + "?", true)) {
            return false;
        }
        this.myPackagesLock.writeLock().lock();
        try {
            Packages pp = this.myPackages.get(pkg);
            if (pp == null) {
                if (force) {
                    boolean bl = false;
                    return bl;
                }
                throw new PackageException("The package does not exist: " + pkg);
            }
            Package p = pp.version(version);
            if (p == null) {
                if (force) {
                    boolean bl = false;
                    return bl;
                }
                throw new PackageException("The version " + version + " does not exist for the package: " + pkg);
            }
            this.myStorage.remove(p);
            pp.remove(p);
            if (pp.latest() == null) {
                this.myPackages.remove(pkg);
            }
        }
        finally {
            this.myPackagesLock.writeLock().unlock();
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Source resolve(String href, URISpace space) throws PackageException {
        LOG.debug("Repository, resolve in {}: '{}'", (Object)space, (Object)href);
        try {
            this.myPackagesLock.readLock().lock();
            for (Packages pp : this.myPackages.values()) {
                Package p = pp.latest();
                Source src = p.resolve(href, space);
                if (src == null) continue;
                Source source = src;
                return source;
            }
        }
        finally {
            this.myPackagesLock.readLock().unlock();
        }
        return null;
    }

    @Override
    public Source resolve(String href, URISpace space, boolean transitive) throws PackageException {
        return this.resolve(href, space);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<PackageException> parsePublicUris() {
        Set<String> packages;
        ArrayList<PackageException> exceptions = null;
        try {
            packages = this.myStorage.listPackageDirectories();
        }
        catch (PackageException e) {
            exceptions = new ArrayList<PackageException>();
            exceptions.add(e);
            return exceptions;
        }
        DescriptorParser parser = new DescriptorParser();
        for (String p : packages) {
            try {
                Storage.PackageResolver res = this.myStorage.makePackageResolver(p, null);
                Source desc = res.resolveResource("expath-pkg.xml");
                Package pkg = parser.parse(desc, p, this.myStorage, this);
                this.addPackage(pkg);
                this.myExtensionsLock.readLock().lock();
                try {
                    for (Extension ext : this.myExtensions.values()) {
                        ext.init(this, pkg);
                    }
                }
                finally {
                    this.myExtensionsLock.readLock().unlock();
                }
            }
            catch (PackageException | Storage.NotExistException e) {
                if (exceptions == null) {
                    exceptions = new ArrayList();
                }
                if (e instanceof Storage.NotExistException) {
                    exceptions.add(new PackageException("Package descriptor does NOT exist in: " + p, e));
                    continue;
                }
                exceptions.add((PackageException)e);
            }
        }
        if (exceptions != null) {
            return exceptions;
        }
        return Collections.emptyList();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addPackage(Package pkg) {
        String name = pkg.getName();
        this.myPackagesLock.writeLock().lock();
        try {
            Packages pp = this.myPackages.get(name);
            if (pp == null) {
                pp = new Packages(name);
                this.myPackages.put(name, pp);
            }
            pp.add(pkg);
        }
        finally {
            this.myPackagesLock.writeLock().unlock();
        }
    }

    Repository() {
        this.myStorage = null;
    }

    public static class HttpException
    extends OnlineException {
        private final int myCode;
        private final String myStatus;

        public HttpException(URI url, int code, String status) {
            super(url, "HTTP error at URL: " + url + ", code: " + code + ", status: " + status);
            this.myCode = code;
            this.myStatus = status;
        }

        public int getCode() {
            return this.myCode;
        }

        public String getStatus() {
            return this.myStatus;
        }
    }

    public static class NotFoundException
    extends OnlineException {
        public NotFoundException(URI url) {
            super(url, "Package not found at URL: " + url);
        }
    }

    public static class OnlineException
    extends PackageException {
        private final URI myUrl;

        public OnlineException(URI url) {
            super("Error downloading the package at URL: " + url);
            this.myUrl = url;
        }

        public OnlineException(URI url, String msg) {
            super(msg);
            this.myUrl = url;
        }

        public OnlineException(URI url, Exception cause) {
            super("Error downloading the package at URL: " + url, cause);
            this.myUrl = url;
        }

        public URI getUrl() {
            return this.myUrl;
        }
    }

    public static class AlreadyInstalledException
    extends PackageException {
        private final String myName;
        private final String myVersion;

        public AlreadyInstalledException(String name, String version) {
            super("Same version of the package is already installed: " + name + ", " + version);
            this.myName = name;
            this.myVersion = version;
        }

        public String getName() {
            return this.myName;
        }

        public String getVersion() {
            return this.myVersion;
        }
    }
}

