package com.apple.foundationdb;

/**
 * A set of options that can be set on a {@link Database}.
 */
public class DatabaseOptions extends OptionsSet {
	public DatabaseOptions( OptionConsumer consumer ) { super(consumer); }

	/**
	 * Set the size of the client location cache. Raising this value can boost performance in very large databases where clients access data in a near-random pattern. Defaults to 100000.
	 * 
	 * @param value Max location cache entries
	 */
	public void setLocationCacheSize(long value) { setOption(10, value); }

	/**
	 * Set the maximum number of watches allowed to be outstanding on a database connection. Increasing this number could result in increased resource usage. Reducing this number will not cancel any outstanding watches. Defaults to 10000 and cannot be larger than 1000000.
	 * 
	 * @param value Max outstanding watches
	 */
	public void setMaxWatches(long value) { setOption(20, value); }

	/**
	 * Specify the machine ID that was passed to fdbserver processes running on the same machine as this client, for better location-aware load balancing.
	 * 
	 * @param value Hexadecimal ID
	 */
	public void setMachineId(String value) { setOption(21, value); }

	/**
	 * Specify the datacenter ID that was passed to fdbserver processes running in the same datacenter as this client, for better location-aware load balancing.
	 * 
	 * @param value Hexadecimal ID
	 */
	public void setDatacenterId(String value) { setOption(22, value); }

	/**
	 * Snapshot read operations will see the results of writes done in the same transaction. This is the default behavior.
	 */
	public void setSnapshotRywEnable() { setOption(26); }

	/**
	 * Snapshot read operations will not see the results of writes done in the same transaction. This was the default behavior prior to API version 300.
	 */
	public void setSnapshotRywDisable() { setOption(27); }

	/**
	 * Sets the maximum escaped length of key and value fields to be logged to the trace file via the LOG_TRANSACTION option. This sets the {@code transaction_logging_max_field_length} option of each transaction created by this database. See the transaction option description for more information.
	 * 
	 * @param value Maximum length of escaped key and value fields.
	 */
	public void setTransactionLoggingMaxFieldLength(long value) { setOption(405, value); }

	/**
	 * Set a timeout in milliseconds which, when elapsed, will cause each transaction automatically to be cancelled. This sets the {@code timeout} option of each transaction created by this database. See the transaction option description for more information. Using this option requires that the API version is 610 or higher.
	 * 
	 * @param value value in milliseconds of timeout
	 */
	public void setTransactionTimeout(long value) { setOption(500, value); }

	/**
	 * Set a maximum number of retries after which additional calls to {@code onError} will throw the most recently seen error code. This sets the {@code retry_limit} option of each transaction created by this database. See the transaction option description for more information.
	 * 
	 * @param value number of times to retry
	 */
	public void setTransactionRetryLimit(long value) { setOption(501, value); }

	/**
	 * Set the maximum amount of backoff delay incurred in the call to {@code onError} if the error is retryable. This sets the {@code max_retry_delay} option of each transaction created by this database. See the transaction option description for more information.
	 * 
	 * @param value value in milliseconds of maximum delay
	 */
	public void setTransactionMaxRetryDelay(long value) { setOption(502, value); }

	/**
	 * Set the maximum transaction size in bytes. This sets the {@code size_limit} option on each transaction created by this database. See the transaction option description for more information.
	 * 
	 * @param value value in bytes
	 */
	public void setTransactionSizeLimit(long value) { setOption(503, value); }

	/**
	 * The read version will be committed, and usually will be the latest committed, but might not be the latest committed in the event of a simultaneous fault and misbehaving clock.
	 */
	public void setTransactionCausalReadRisky() { setOption(504); }

	/**
	 * Deprecated. Addresses returned by get_addresses_for_key include the port when enabled. As of api version 630, this option is enabled by default and setting this has no effect.
	 */
	@Deprecated
	public void setTransactionIncludePortInAddress() { setOption(505); }

	/**
	 * Set a random idempotency id for all transactions. See the transaction option description for more information. This feature is in development and not ready for general use.
	 */
	public void setTransactionAutomaticIdempotency() { setOption(506); }

	/**
	 * Allows {@code get} operations to read from sections of keyspace that have become unreadable because of versionstamp operations. This sets the {@code bypass_unreadable} option of each transaction created by this database. See the transaction option description for more information.
	 */
	public void setTransactionBypassUnreadable() { setOption(700); }

	/**
	 * By default, operations that are performed on a transaction while it is being committed will not only fail themselves, but they will attempt to fail other in-flight operations (such as the commit) as well. This behavior is intended to help developers discover situations where operations could be unintentionally executed after the transaction has been reset. Setting this option removes that protection, causing only the offending operation to fail.
	 */
	public void setTransactionUsedDuringCommitProtectionDisable() { setOption(701); }

	/**
	 * Enables conflicting key reporting on all transactions, allowing them to retrieve the keys that are conflicting with other transactions.
	 */
	public void setTransactionReportConflictingKeys() { setOption(702); }

	/**
	 * Use configuration database.
	 */
	public void setUseConfigDatabase() { setOption(800); }

	/**
	 * Enables verification of causal read risky by checking whether clients are able to read stale data when they detect a recovery, and logging an error if so.
	 * 
	 * @param value integer between 0 and 100 expressing the probability a client will verify it can't read stale data
	 */
	public void setTestCausalReadRisky(long value) { setOption(900, value); }
}
