/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.python.embedding.tools.vfs;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Consumer;
import java.util.stream.Stream;
import org.graalvm.python.embedding.tools.exec.GraalPyRunner;
import org.graalvm.python.embedding.tools.exec.SubprocessLog;

public final class VFSUtils {
    public static final String VFS_ROOT = "org.graalvm.python.vfs";
    public static final String VFS_VENV = "venv";
    public static final String VFS_FILESLIST = "fileslist.txt";
    public static final String GRAALPY_GROUP_ID = "org.graalvm.python";
    private static final String NATIVE_IMAGE_RESOURCES_CONFIG = "{\n  \"resources\": {\n    \"includes\": [\n      {\"pattern\": \"$vfs/.*\"}\n    ]\n  }\n}\n";
    private static final boolean IS_WINDOWS = System.getProperty("os.name").startsWith("Windows");
    public static final String LAUNCHER_NAME = IS_WINDOWS ? "graalpy.exe" : "graalpy.sh";
    private static final String GRAALPY_MAIN_CLASS = "com.oracle.graal.python.shell.GraalPythonMain";
    private static final boolean REPLACE_BACKSLASHES = File.separatorChar == '\\';

    public static void writeNativeImageConfig(Path metaInfRoot, String pluginId) throws IOException {
        VFSUtils.writeNativeImageConfig(metaInfRoot, pluginId, VFS_ROOT);
    }

    public static void writeNativeImageConfig(Path metaInfRoot, String pluginId, String vfsRoot) throws IOException {
        Path p = metaInfRoot.resolve(Path.of("native-image", GRAALPY_GROUP_ID, pluginId));
        VFSUtils.write(p.resolve("resource-config.json"), NATIVE_IMAGE_RESOURCES_CONFIG.replace("$vfs", vfsRoot));
    }

    private static void write(Path config, String txt) throws IOException {
        try {
            VFSUtils.createParentDirectories(config);
            Files.writeString(config, (CharSequence)txt, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        }
        catch (IOException e) {
            throw new IOException(String.format("failed to write %s", config), e);
        }
    }

    private static void createParentDirectories(Path path) throws IOException {
        Path parent = path.getParent();
        if (parent != null) {
            Files.createDirectories(parent, new FileAttribute[0]);
        }
    }

    public static void generateVFSFilesList(Path resourcesRoot, Path vfs) throws IOException {
        TreeSet<String> entriesSorted = new TreeSet<String>();
        VFSUtils.generateVFSFilesList(resourcesRoot, vfs, entriesSorted, null);
        Path filesList = vfs.resolve(VFS_FILESLIST);
        Files.write(filesList, entriesSorted, new OpenOption[0]);
    }

    public static void generateVFSFilesList(Path vfs) throws IOException {
        VFSUtils.generateVFSFilesList(null, vfs);
    }

    private static String normalizeResourcePath(String path) {
        return REPLACE_BACKSLASHES ? path.replace("\\", "/") : path;
    }

    public static void generateVFSFilesList(Path vfs, Set<String> ret, Consumer<String> duplicateHandler) throws IOException {
        VFSUtils.generateVFSFilesList(null, vfs, ret, duplicateHandler);
    }

    public static void generateVFSFilesList(Path resourcesRoot, Path vfs, Set<String> ret, Consumer<String> duplicateHandler) throws IOException {
        int rootEndIdx;
        if (!Files.isDirectory(vfs, new LinkOption[0])) {
            throw new IOException(String.format("'%s' has to exist and be a directory.\n", vfs));
        }
        String rootPath = VFSUtils.makeDirPath(vfs.toAbsolutePath());
        if (resourcesRoot == null) {
            rootEndIdx = rootPath.lastIndexOf(File.separator, rootPath.lastIndexOf(File.separator) - 1);
        } else {
            String resRootPath = VFSUtils.makeDirPath(resourcesRoot);
            rootEndIdx = resRootPath.length() - 1;
        }
        try (Stream<Path> s = Files.walk(vfs, new FileVisitOption[0]);){
            s.forEach(p -> {
                String entry = null;
                if (Files.isDirectory(p, new LinkOption[0])) {
                    String dirPath = VFSUtils.makeDirPath(p.toAbsolutePath());
                    entry = dirPath.substring(rootEndIdx);
                } else if (Files.isRegularFile(p, new LinkOption[0])) {
                    entry = p.toAbsolutePath().toString().substring(rootEndIdx);
                }
                if (entry != null && !ret.add(entry = VFSUtils.normalizeResourcePath(entry)) && duplicateHandler != null) {
                    duplicateHandler.accept(entry);
                }
            });
        }
    }

    private static String makeDirPath(Path p) {
        Object ret = p.toString();
        if (!((String)ret).endsWith(File.separator)) {
            ret = (String)ret + File.separator;
        }
        return ret;
    }

    public static void delete(Path dir) throws IOException {
        if (!Files.exists(dir, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> s = Files.walk(dir, new FileVisitOption[0]);){
            s.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
        }
        catch (IOException e) {
            throw new IOException(String.format("failed to delete %s", dir), e);
        }
    }

    public static void createVenv(Path venvDirectory, List<String> packages, Path launcher, LauncherClassPath launcherClassPath, String graalPyVersion, SubprocessLog subprocessLog, Log log) throws IOException {
        Path launcherPath = launcher;
        String externalLauncher = System.getProperty("graalpy.vfs.venvLauncher");
        if (externalLauncher == null || externalLauncher.trim().isEmpty()) {
            VFSUtils.generateLaunchers(launcherPath, launcherClassPath, subprocessLog, log);
        } else {
            launcherPath = Path.of(externalLauncher, new String[0]);
        }
        if (packages != null) {
            VFSUtils.trim(packages);
        }
        ArrayList<String> installedPackages = new ArrayList<String>();
        Path tag = venvDirectory.resolve("contents");
        if (Files.exists(venvDirectory, new LinkOption[0])) {
            VFSUtils.checkLauncher(venvDirectory, launcherPath, log);
            if (Files.isReadable(tag)) {
                List<String> lines = null;
                try {
                    lines = Files.readAllLines(tag);
                }
                catch (IOException e) {
                    throw new IOException(String.format("failed to read tag file %s", tag), e);
                }
                if (lines.isEmpty() || !graalPyVersion.equals(lines.get(0))) {
                    log.info(String.format("Stale GraalPy venv, updating to %s", graalPyVersion));
                    VFSUtils.delete(venvDirectory);
                } else {
                    for (int i = 1; i < lines.size(); ++i) {
                        installedPackages.add(lines.get(i));
                    }
                }
            }
        }
        if (!Files.exists(venvDirectory, new LinkOption[0])) {
            log.info(String.format("Creating GraalPy %s venv", graalPyVersion));
            VFSUtils.runLauncher(launcherPath.toString(), subprocessLog, "-m", VFS_VENV, venvDirectory.toString(), "--without-pip");
            VFSUtils.runVenvBin(venvDirectory, "graalpy", subprocessLog, "-I", "-m", "ensurepip");
        }
        List<String> frozenPkgs = null;
        if (packages != null) {
            boolean needsUpdate = false;
            needsUpdate |= VFSUtils.deleteUnwantedPackages(venvDirectory, packages, installedPackages, subprocessLog);
            if (needsUpdate |= VFSUtils.installWantedPackages(venvDirectory, packages, installedPackages, subprocessLog)) {
                SubprocessLog.CollectOutputLog freezeLog = new SubprocessLog.CollectOutputLog();
                VFSUtils.runPip(venvDirectory, "freeze", freezeLog, "--local");
                frozenPkgs = freezeLog.getOutput();
            }
        }
        try {
            Files.write(tag, List.of(graalPyVersion), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
            Files.write(tag, packages, StandardOpenOption.APPEND);
            if (frozenPkgs != null) {
                String toWrite = "# Generated by GraalPy Maven or Gradle plugin using pip freeze\n# This file is used by GraalPy VirtualFileSystem\n" + String.join((CharSequence)"\n", frozenPkgs);
                Files.write(venvDirectory.resolve("installed.txt"), toWrite.getBytes(StandardCharsets.UTF_8), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
            }
        }
        catch (IOException e) {
            throw new IOException(String.format("failed to write tag file %s", tag), e);
        }
    }

    private static void checkLauncher(Path venvDirectory, Path launcherPath, Log log) throws IOException {
        if (!Files.exists(launcherPath, new LinkOption[0])) {
            throw new IOException(String.format("Launcher file does not exist '%s'", launcherPath));
        }
        Path cfg = venvDirectory.resolve("pyvenv.cfg");
        if (Files.exists(cfg, new LinkOption[0])) {
            try {
                List<String> lines = Files.readAllLines(cfg);
                for (String line : lines) {
                    int idx = line.indexOf("=");
                    if (idx <= -1) continue;
                    String l = line.substring(0, idx).trim();
                    String r = line.substring(idx + 1).trim();
                    if (!l.trim().equals("executable")) continue;
                    Path cfgLauncherPath = Path.of(r, new String[0]);
                    if (Files.exists(cfgLauncherPath, new LinkOption[0]) && Files.isSameFile(launcherPath, cfgLauncherPath)) break;
                    log.info(String.format("Deleting GraalPy venv due to changed launcher path", new Object[0]));
                    VFSUtils.delete(venvDirectory);
                }
            }
            catch (IOException e) {
                e.printStackTrace();
                throw new IOException(String.format("failed to read config file %s", cfg), e);
            }
        } else {
            log.info(String.format("missing venv config file: '%s'", cfg));
        }
    }

    private static void generateLaunchers(Path laucherPath, LauncherClassPath launcherClassPath, SubprocessLog subprocessLog, Log log) throws IOException {
        if (!Files.exists(laucherPath, new LinkOption[0])) {
            File tmp;
            log.info("Generating GraalPy launchers");
            VFSUtils.createParentDirectories(laucherPath);
            Path java = Paths.get(System.getProperty("java.home"), "bin", "java");
            String classpath = String.join((CharSequence)File.pathSeparator, launcherClassPath.get());
            if (!IS_WINDOWS) {
                String script = String.format("#!/usr/bin/env bash\n%s --enable-native-access=ALL-UNNAMED -classpath %s %s --python.Executable=\"$0\" \"$@\"\n", java, String.join((CharSequence)File.pathSeparator, classpath), GRAALPY_MAIN_CLASS);
                try {
                    Files.writeString(laucherPath, (CharSequence)script, new OpenOption[0]);
                    Set<PosixFilePermission> perms = Files.getPosixFilePermissions(laucherPath, new LinkOption[0]);
                    perms.addAll(List.of(PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_EXECUTE));
                    Files.setPosixFilePermissions(laucherPath, perms);
                }
                catch (IOException e) {
                    throw new IOException(String.format("failed to create launcher %s", laucherPath), e);
                }
            }
            String script = String.format("import os, shutil, struct, venv\nfrom pathlib import Path\nvl = os.path.join(venv.__path__[0], 'scripts', 'nt', 'graalpy.exe')\ntl = os.path.join(r'%s')\nos.makedirs(Path(tl).parent.absolute(), exist_ok=True)\nshutil.copy(vl, tl)\ncmd = r'%s --enable-native-access=ALL-UNNAMED -classpath \"%s\" %s'\npyvenvcfg = os.path.join(os.path.dirname(tl), \"pyvenv.cfg\")\nwith open(pyvenvcfg, 'w', encoding='utf-8') as f:\n    f.write('venvlauncher_command = ')\n    f.write(cmd)\n", laucherPath, java, classpath, GRAALPY_MAIN_CLASS);
            try {
                tmp = File.createTempFile("create_launcher", ".py");
            }
            catch (IOException e) {
                throw new IOException("failed to create tmp launcher", e);
            }
            tmp.deleteOnExit();
            try (FileWriter wr = new FileWriter(tmp);){
                wr.write(script);
            }
            catch (IOException e) {
                throw new IOException(String.format("failed to write tmp launcher %s", tmp), e);
            }
            try {
                GraalPyRunner.run(classpath, subprocessLog, tmp.getAbsolutePath());
            }
            catch (InterruptedException e) {
                throw new IOException(String.format("failed to run Graalpy launcher", new Object[0]), e);
            }
        }
    }

    private static boolean installWantedPackages(Path venvDirectory, List<String> packages, List<String> installedPackages, SubprocessLog subprocessLog) throws IOException {
        HashSet<String> pkgsToInstall = new HashSet<String>(packages);
        pkgsToInstall.removeAll(installedPackages);
        if (pkgsToInstall.isEmpty()) {
            return false;
        }
        VFSUtils.runPip(venvDirectory, "install", subprocessLog, pkgsToInstall.toArray(new String[pkgsToInstall.size()]));
        return true;
    }

    private static boolean deleteUnwantedPackages(Path venvDirectory, List<String> packages, List<String> installedPackages, SubprocessLog subprocessLog) throws IOException {
        ArrayList<String> args = new ArrayList<String>(installedPackages);
        args.removeAll(packages);
        if (args.isEmpty()) {
            return false;
        }
        args.add(0, "-y");
        VFSUtils.runPip(venvDirectory, "uninstall", subprocessLog, args.toArray(new String[args.size()]));
        return true;
    }

    private static void runLauncher(String launcherPath, SubprocessLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runLauncher(launcherPath, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute launcher command %s", List.of(args)));
        }
    }

    private static void runPip(Path venvDirectory, String command, SubprocessLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runPip(venvDirectory, command, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute pip %s", List.of(args)), e);
        }
    }

    private static void runVenvBin(Path venvDirectory, String bin, SubprocessLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runVenvBin(venvDirectory, bin, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute venv %s", List.of(args)), e);
        }
    }

    public static List<String> trim(List<String> l) {
        Iterator<String> it = l.iterator();
        while (it.hasNext()) {
            String p = it.next();
            if (p != null && !p.trim().isEmpty()) continue;
            it.remove();
        }
        return l;
    }

    @FunctionalInterface
    public static interface LauncherClassPath {
        public Set<String> get() throws IOException;
    }

    public static interface Log {
        public void info(String var1);
    }
}

