package tech.beepbeep.beep_loader

import bolts.Task
import com.google.gson.JsonObject
import com.google.gson.JsonParser
import tech.beepbeep.beep_commons.util.Constants
import tech.beepbeep.beep_commons.util.MessageBody
import tech.beepbeep.beep_commons.util.MessageProcessor
import tech.beepbeep.beep_commons.util.Utils.Companion.printLog
import java.io.*
import java.net.HttpURLConnection
import java.net.URL
import java.security.MessageDigest
import kotlin.Exception

/**
 * This class be used to download first core jar if the app first time implement beep middle-ware
 *
 *  Support java
 *
 * @constructor Create empty Core downloader
 */
class CoreDownloader {

    var bcURL = ""
    var coreURL = ""
    var bcChecksum = ""
    var coreChecksum = ""

    /**
     * Download initial Core.jar when first time implementing beep middle ware.
     * there's no reason to use Retrofit here. We're not leveraging any of its features
     * and it will actually make the download slower since the entire response body will be
     * buffered in memory before being written out to the filesystem. If we use OkHttp directly
     * we can stream the bytes directly to the filesystem.
     *
     * @param path                      save the file to this path, should be same as the log path
     * @param url                       download url
     * @param messageProcessor          callback to update download process to external party
     */
    private fun downloadCoreFromUrl(path: String, url: String, messageProcessor: MessageProcessor) {
        messageProcessor.processMessage(
                MessageBody(
                        what = Constants.Core.JAR_DOWNLOADING.value,
                        obj1 = Constants.INITIAL_CORE
                )
        )
        val jarUrl = URL(url)
        var urlConnection: HttpURLConnection? = null
        try {
            printLog("$TAG, Try to Get File Size")
            urlConnection = jarUrl
                .openConnection() as HttpURLConnection
            urlConnection.requestMethod = "HEAD"
            val inputStreamForHead: InputStream = urlConnection.inputStream
            printLog("$TAG, File Size Is: ${(urlConnection.contentLength/1024)} KB")
            inputStreamForHead.close()
            urlConnection.disconnect()
        } catch (e: Exception) {
            printLog("$TAG, Get Size fail Cos: $e")
            urlConnection?.disconnect()
        }
        try {
            val fileName: String = url.substring( url.lastIndexOf('/')+1, url.length)
            BufferedInputStream(
                URL(url).openStream()
            ).use { inputStream ->
                printLog("$TAG, Reading $fileName...")
                FileOutputStream("$path${File.separator}${fileName}").use { fileOS ->
                    val data = ByteArray(1024)
                    var byteContent: Int
                    val digest = MessageDigest.getInstance("MD5")
                    while (inputStream.read(data, 0, 1024).also { byteContent = it } != -1) {
                        fileOS.write(data, 0, byteContent)
                        digest.update(data.copyOfRange(0, byteContent))
                    }
                    fileOS.flush()
                    fileOS.close()
                    inputStream.close()
                    printLog("$TAG, Download Initial Core Finish")
                    if (coreChecksum.isNotEmpty() && !coreChecksum.equals(LoaderUtils.bytesToHex(digest.digest()).also {
                            printLog("$TAG, checksum expected = $coreChecksum, actual = $it")
                        }, ignoreCase = true)) {
                        printLog("$TAG, Wrong checksum, Retry After 60s")
                        File("$path${File.separator}${fileName}").delete()
                        Task.delay(60000L).continueWith {
                            downloadCoreFromUrl(path,url,messageProcessor)
                        }
                        return
                    }
                    messageProcessor.processMessage(
                        MessageBody(
                            what = Constants.Core.CORE_JAR_DOWNLOAD_DONE.value,
                            obj1 =  fileName
                        )
                    )
                }
            }
        } catch (e: IOException) {
            printLog("$TAG, $e Retry After 60s")
            Task.delay(60000L).continueWith {
                downloadCoreFromUrl(path,url,messageProcessor)
            }
        }
    }

    private fun downloadBcFromURL(path: String, url: String, messageProcessor: MessageProcessor) {
        messageProcessor.processMessage(
            MessageBody(
                what = Constants.Core.JAR_DOWNLOADING.value,
                obj1 = BeepLoaderManager.BOUNCY_CASTLE
            )
        )
        val jarUrl = URL(url)
        var urlConnection: HttpURLConnection? = null
        try {
            printLog("$TAG, Try to Get File Size")
            urlConnection = jarUrl
                .openConnection() as HttpURLConnection
            urlConnection.requestMethod = "HEAD"
            val inputStreamForHead: InputStream = urlConnection.inputStream
            printLog("$TAG, File Size Is: ${(urlConnection.contentLength/1024)} KB")
            inputStreamForHead.close()
            urlConnection.disconnect()
        } catch (e: Exception) {
            printLog("$TAG, Get Size fail Cos: $e")
            urlConnection?.disconnect()
        }
        try {
            val fileName: String = url.substring( url.lastIndexOf('/')+1, url.length)
            BufferedInputStream(
                URL(url).openStream()
            ).use { inputStream ->
                printLog("$TAG, Reading $fileName...")
                FileOutputStream("$path${File.separator}${fileName}").use { fileOS ->
                    val data = ByteArray(1024)
                    var byteContent: Int
                    val digest = MessageDigest.getInstance("MD5")
                    while (inputStream.read(data, 0, 1024).also { byteContent = it } != -1) {
                        fileOS.write(data, 0, byteContent)
                        digest.update(data.copyOfRange(0, byteContent))
                    }
                    fileOS.flush()
                    fileOS.close()
                    inputStream.close()
                    printLog("$TAG, Download $fileName Done")
                    if (bcChecksum.isNotEmpty() && !bcChecksum.equals(LoaderUtils.bytesToHex(digest.digest()).also {
                            printLog("$TAG, checksum expected = $bcChecksum, actual = $it")
                        }, ignoreCase = true)) {
                        printLog("$TAG, Wrong checksum, Retry After 60s")
                        File("$path${File.separator}${fileName}").delete()
                        Task.delay(60000L).continueWith {
                            downloadCoreFromUrl(path,url,messageProcessor)
                        }
                        return
                    }

                    downloadCoreFromUrl(path, coreURL, messageProcessor)
                }
            }
        } catch (e: IOException) {
            printLog("$TAG, $e Retry After 60s")
            Task.delay(60000L).continueWith {
                downloadBcFromURL(path,url,messageProcessor)
            }
        }
    }

    /**
     * Sent http request to get the first core jar file download link
     *
     * @param path                 save the file to this path, should be same as the log path
     * @param messageProcessor     callback to update download process to external party
     */
     fun getDownloadLink(path: String, messageProcessor: MessageProcessor) {
        if (path.isNullOrEmpty()) {
            printLog("$TAG, Cannot perform download, please set log path first")
            return
        }
//        messageProcessor.processMessage(
//                MessageBody(
//                        what = Constants.Core.JAR_DOWNLOADING.value,
//                        obj1 = Constants.INITIAL_CORE
//                )
//        )
        val url: URL
        var urlConnection: HttpURLConnection? = null
        try {
            printLog("$TAG, Try to get download Link")

            url = if(BeepLoaderManager.getInstance().isDebug) URL(Constants.DEV_CORE_URL) else URL(Constants.LIVE_CORE_URL)
            urlConnection = url
                    .openConnection() as HttpURLConnection
            urlConnection.requestMethod = "GET"
            //urlConnection.setRequestProperty("action,")
            val inputStream: InputStream = urlConnection.inputStream
            val inputStreamReader = InputStreamReader(inputStream)
            val reader = BufferedReader(inputStreamReader)
            val sb = StringBuffer()

            var str: String?
            while (reader.readLine().also { str = it } != null) {
                sb.append(str)
            }
            printLog("$TAG, $sb")
            inputStream.close()

            val myJsonObject = JsonParser().parse(sb.toString()) as JsonObject
            val data = myJsonObject.get("data")
            val myUrlObject = JsonParser().parse(data.toString()) as JsonObject
            coreURL = myUrlObject.get("java").asString
            coreChecksum = myUrlObject.get("java_checksum").asString
            bcURL = myUrlObject.get("bc").asString
            bcChecksum = myUrlObject.get("bc_checksum").asString

            downloadBcFromURL(path,bcURL,messageProcessor)
        } catch (e: java.lang.Exception) {
            printLog("$TAG, Request Initial Core Download Link Failure: $e Retry After 60 Seconds")
            Task.delay(60000L).continueWith {
                getDownloadLink(path,messageProcessor)
            }
        } finally {
            urlConnection?.disconnect()
        }
    }


    companion object {
        private val TAG = CoreDownloader::class.java.simpleName
    }
}