/*
 * Copyright (c) 2019. NUM Technology Ltd
 */

package uk.num.numlib.internal.module;

import lombok.extern.log4j.Log4j2;
import uk.num.numlib.exc.NumBadURLException;
import uk.num.numlib.exc.NumInvalidParameterException;
import uk.num.numlib.internal.ctx.AppContext;
import uk.num.numlib.internal.util.NonBlankString;
import uk.num.numlib.internal.util.SimpleCache;

/**
 * A factory for ModuleDNSQuery objects.
 *
 * @author tonywalmsley
 */
@Log4j2
public class ModuleFactory {
    /**
     * A cache for module/NUM ID combinations.
     */
    private final SimpleCache<String, ModuleDNSQueries> moduleMap = new SimpleCache<>();

    /**
     * Create and initialise a ModuleDNSQueries object or use a cached object.
     *
     * @param appContext the AppContext
     * @param moduleName the module name string, e.g. "1"
     * @param numId      the NUM ID to be queried for a NUM record.
     * @return a ModuleDNSQueries object
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public ModuleDNSQueries getInstance(final AppContext appContext, final NonBlankString moduleName, final NonBlankString numId) throws
                                                                                                                                  NumInvalidParameterException,
                                                                                                                                  NumBadURLException {
        ModuleDNSQueries result;

        final String key = moduleName.value + "_" + numId.value;
        // Critical section - we're reading then updating moduleMap, which is a potential race condition
        synchronized (moduleMap) {
            result = moduleMap.get(key);
            if (result == null) {
                result = new ModuleDNSQueries(moduleName, numId);

                // Initialisation as a separate step since its an 'expensive' operation. Allows us to create lots of
                // Modules if necessary but then only initialise the ones we use.
                result.initialise(appContext);

                // Do this last in case there's an exception so we don't store an invalid ModuleDNSQueries object
                moduleMap.put(key, result);
                log.trace("Cached a new set of ModuleDNSQueries.");
            } else {
                log.trace("Using cached ModuleDNSQueries.");
            }
        }
        return result;
    }
}
