package fi.evolver.script;

import java.io.*;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Stream;

public class Shell {
	private static final ExecutorService EXECUTOR = Executors.newCachedThreadPool(r -> {
		Thread thread = Executors.defaultThreadFactory().newThread(r);
		thread.setDaemon(true);
		return thread;
	});

	public static final Path HOME = Path.of(System.getProperty("user.home"));
	public static final Path BASHRC = HOME.resolve(".bashrc");
	public static final String USER = System.getProperty("user.name");

	private static Writer globalLogWriter;

	private static String password;

	public static void setGlobalLogWriter(Writer globalLogWriter) {
		Shell.globalLogWriter = globalLogWriter;
	}

	/**
	 * Run an external command as the current user.
	 * Fails on non-zero exit value.
	 *
	 * @param command The command to run.
	 * @return Standard output returned by the program.
	 */
	public static String user(String... command) {
		return user(Arrays.asList(command));
	}

	/**
	 * Run an external command as the current user.
	 * Fails on non-zero exit value.
	 *
	 * @param command The command to run.
	 * @return Standard output returned by the program.
	 */
	public static String user(List<String> command) {
		return Command.user(command).run().stdout();
	}


	/**
	 * Run an external command with root privileges.
	 * Fails on non-zero exit value.
	 *
	 * @param command The command to run.
	 * @return Standard output returned by the program.
	 */
	public static String sudo(String... command) {
		return sudo(Arrays.asList(command));
	}

	/**
	 * Run an external command with root privileges.
	 * Fails on non-zero exit value.
	 *
	 * @param command The command to run.
	 * @return Standard output returned by the program.
	 */
	public static String sudo(List<String> command) {
		return Command.sudo(command).run().stdout();
	}


	private static String readTextStream(InputStream inputStream, PrintStream output) throws IOException {
		char[] buffer = new char[8 * 1024];
		StringBuilder builder = new StringBuilder();
		try (Reader reader = new InputStreamReader(new BufferedInputStream(inputStream))) {
			int count;
			while ((count = reader.read(buffer)) != -1) {
				builder.append(buffer, 0, count);
				if (Step.DEBUG_ENABLED && output != null) {
					output.print(new String(buffer, 0, count));
					output.flush();
				}
				if (globalLogWriter != null) {
					globalLogWriter.write(buffer, 0, count);
					globalLogWriter.flush();
				}
			}
		}
		return builder.toString();
	}

	private static byte[] readBinaryStream(InputStream inputStream) throws IOException {
		ByteArrayOutputStream collector = new ByteArrayOutputStream();
		inputStream.transferTo(collector);
		return collector.toByteArray();
	}

	public static class Command {
		private final List<String> command;
		private final Map<String, String> environment = new LinkedHashMap<>();
		private Optional<InputStream> stdin = Optional.empty();
		private boolean failOnError = true;
		private Path workingDirectory;
		private boolean binaryStdout;

		private Command(List<String> command) {
			this.command = command;
		}


		/**
		 * Set an environment variable for the command.
		 *
		 * @param variable The variable name.
		 * @param value The value for the variable.
		 * @return This command for chaining purposes.
		 */
		public Command env(String variable, String value) {
			environment.put(variable, value);
			return this;
		}

		/**
		 * Add standard input for the command.
		 *
		 * @param data Data to add to the standard input stream.
		 * @return This command for chaining purposes.
		 */
		public Command stdin(InputStream data) {
			stdin = Stream.concat(stdin.stream(), Optional.ofNullable(data).stream())
					.reduce(SequenceInputStream::new);
			return this;
		}

		/**
		 * Add standard input for the command.
		 *
		 * @param data Data to add to the standard input stream.
		 * @return This command for chaining purposes.
		 */
		public Command stdin(String data) {
			return stdin(new ByteArrayInputStream(data.getBytes()));
		}


		/**
		 * Sets whether the command should fail on non-zero exit value.
		 *
		 * @param value Should the command fail on error.
		 * @return This command for chaining purposes.
		 */
		public Command failOnError(boolean value) {
			this.failOnError = value;
			return this;
		}

		public Command workingDirectory(Path path) {
			this.workingDirectory = path;
			return this;
		}

		public Command binaryStdout(boolean value) {
			this.binaryStdout = value;
			return this;
		}

		/**
		 * Is this command run with root privileges.
		 *
		 * @return Whether this command uses sudo.
		 */
		public boolean isSudo() {
			return "sudo".equals(command.stream().findFirst().orElse("-"));
		}


		@Override
		public String toString() {
			StringBuilder builder = new StringBuilder();
			builder.append(isSudo() ? '#' : '$');
			command.forEach(p -> builder.append(" ").append(p));
			return builder.toString();
		}


		/**
		 * Create a new command run with the current user's privileges.
		 *
		 * @param command The command to run.
		 * @return The created command.
		 */
		public static Command user(String... command) {
			return user(Arrays.asList(command));
		}

		/**
		 * Create a new command run with the current user's privileges.
		 *
		 * @param command The command to run.
		 * @return The created command.
		 */
		public static Command user(List<String> command) {
			return new Command(command);
		}


		/**
		 * Create a new command run with super user privileges.
		 *
		 * @param command The command to run.
		 * @return The created command.
		 */
		public static Command sudo(String... command) {
			return sudo(Arrays.asList(command));
		}

		/**
		 * Create a new command run with super user privileges.
		 *
		 * @param command The command to run.
		 * @return The created command.
		 */
		public static Command sudo(List<String> command) {
			if ("root".equals(System.getProperty("user.name"))) {
				return user(command);
			}

			if (password == null)
				password = Dialog.readPassword("Please input user password for using sudo");

			List<String> parts = new ArrayList<>();
			parts.addAll(List.of("sudo", "-S", "-k", "-p", ""));
			parts.addAll(command);
			return new Command(parts)
					.stdin(password + "\n");
		}


		/**
		 * Execute the command.
		 *
		 * @return The results of the command.
		 */
		public Result run() {
			try (Step step = Step.start(toString())) {
				ProcessBuilder builder = new ProcessBuilder();
				builder.command(command);
				builder.environment().putAll(environment);
				if (workingDirectory != null)
					builder.directory(workingDirectory.toFile());

				if (globalLogWriter != null) {
					globalLogWriter.write("\n\n%s Running command: %s\n".formatted(LocalDateTime.now(), this));
				}

				Process process = builder.start();

				Future<String> futureStdoutText = binaryStdout
						? CompletableFuture.completedFuture(null)
						: EXECUTOR.submit(() -> readTextStream(process.getInputStream(), System.out));

				Future<byte[]> futureStdoutBytes = binaryStdout
						? EXECUTOR.submit(() -> readBinaryStream(process.getInputStream()))
						: CompletableFuture.completedFuture(null);

				Future<String> futureStderr = EXECUTOR.submit(() -> readTextStream(process.getErrorStream(), System.err));

				if (stdin.isPresent()) {
					try (OutputStream out = new BufferedOutputStream(process.getOutputStream())) {
						stdin.get().transferTo(out);
					}
				}

				int exitValue = process.waitFor();

				Result result = new Result(
						exitValue,
						futureStdoutBytes.get(),
						futureStdoutText.get(),
						futureStderr.get()
				);

				if (failOnError && !result.success()) {
					if (Step.DEBUG_ENABLED) {
						// Already output
						step.fail("FAILED");
					} else {
						step.fail("STDOUT:\n%s\nSTDERR:\n%s".formatted(
								binaryStdout ? "(binary data)" : result.stdout(),
								result.stderr()
						));
					}
				}

				return result;
			} catch (IOException | ExecutionException e) {
				throw new RuntimeException("Failed executing a command", e);
			} catch (InterruptedException e) {
				throw new RuntimeException("Interrupted while executing a command", e);
			}
		}
	}

	/**
	 * @param exitValue the exit value of the command.
	 * @param binaryStdout the raw bytes of the standard output. Only available if the command was run with binaryStdout=true.
	 * @param stdout the standard output as a string. Only available if the command was run with binaryStdout=false (the default).
	 * @param stderr the standard error as a string
	 */
	public record Result(
			int exitValue,
			byte[] binaryStdout,
			String stdout,
			String stderr
	) {
		public boolean success() {
			return exitValue == 0;
		}

	}

}
