package fi.evolver.script.app;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.time.Duration;
import java.util.Base64;
import java.util.Optional;

import fi.evolver.script.Dialog;
import fi.evolver.script.Environment;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

public class Maven {
	private static final Dialog.DialogTextField DIALOG_FIELD_USERNAME = Dialog.textField("username");
	private static final Dialog.DialogTextField DIALOG_FIELD_PASSWORD = Dialog.passwordField("password");

	private static final String SUFFIX_USERNAME = "USERNAME";
	private static final String SUFFIX_PASSWORD = "PASSWORD";

	private static final Duration VALIDATION_TIMEOUT = Duration.ofSeconds(10);

	public static void writeSettings(Path source) throws IOException {
		try (Step step = Step.start("Maven: write settings")) {
			Path mavenSettingsPath = Shell.HOME.resolve(".m2/settings.xml");
			Files.createDirectories(mavenSettingsPath.getParent());
			Files.copy(source, mavenSettingsPath, StandardCopyOption.REPLACE_EXISTING);
		}
	}

	public static void setupMavenCredentials(Repo repo) {
		try (Step step = Step.start("Setup Maven credentials")) {
			if (hasValidCredentials(repo)) {
				step.skip("Already valid");
				return;
			}

			Credentials credentials = readAndValidateCredentials(repo);
			setCredentials(repo, credentials);
		}
	}

	private static Credentials readAndValidateCredentials(Repo repo) {
		Optional<String> errorInfo = Optional.empty();

		while (true) {
			Dialog dialog = new Dialog("Maven credentials: %s".formatted(repo.name))
					.add(DIALOG_FIELD_USERNAME)
					.add(DIALOG_FIELD_PASSWORD);

			errorInfo
					.map("Error: %s"::formatted)
					.map(Dialog::paragraph)
					.ifPresent(dialog::add);

			var dialogResult = dialog.show();

			Credentials credentials = new Credentials(
					dialogResult.get(DIALOG_FIELD_USERNAME),
					dialogResult.get(DIALOG_FIELD_PASSWORD)
			);

			try {
				var validationResponse = validateCredentials(credentials, repo.pingUrl);
				if (validationResponse.statusCode() == 200)
					return credentials;

				errorInfo = Optional.of("%s %s".formatted(validationResponse.statusCode(), validationResponse.body()));
			} catch (IOException e) {
				errorInfo = Optional.of(e.toString());
			}
		}
	}

	private static HttpResponse<String> validateCredentials(Credentials credentials, URI repoPingUrl) throws IOException {
		try (HttpClient client = HttpClient.newBuilder().connectTimeout(VALIDATION_TIMEOUT).build()) {
			String authHeader = "Basic " + Base64.getEncoder().encodeToString(
					(credentials.username + ":" + credentials.password).getBytes());

			HttpRequest request = HttpRequest.newBuilder()
					.uri(repoPingUrl)
					.header("Authorization", authHeader)
					.timeout(VALIDATION_TIMEOUT)
					.GET()
					.build();

			return client.send(request, HttpResponse.BodyHandlers.ofString());
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			throw new RuntimeException("Credential validation interrupted", e);
		}
	}

	private static boolean hasValidCredentials(Repo repo) {
		Optional<Credentials> existingCredentials = readExistingCredentials(repo);
		if (existingCredentials.isEmpty())
			return false;

		try {
			return validateCredentials(existingCredentials.get(), repo.pingUrl).statusCode() == 200;
		} catch (IOException ignore) {
			return false;
		}
	}

	private static Optional<Credentials> readExistingCredentials(Repo repo) {
		Optional<String> username = Environment.getPersonalVariable(repo.usernameVariable());
		Optional<String> password = Environment.getPersonalVariable(repo.passwordVariable());
		if (username.isEmpty() || password.isEmpty())
			return Optional.empty();

		return Optional.of(new Credentials(username.get(), password.get()));
	}


	private static void setCredentials(Repo repo, Credentials credentials) {
		Environment.setPersonalVariable(repo.usernameVariable(), credentials.username);
		Environment.setPersonalVariable(repo.passwordVariable(), credentials.password);
	}


	public static Repo repo(String name, URI pingUrl) {
		return new Repo(name, pingUrl);
	}

	public record Repo(
			String name,
			URI pingUrl) {

		private String usernameVariable() {
			return createVariableKey(SUFFIX_USERNAME);
		}

		private String passwordVariable() {
			return createVariableKey(SUFFIX_PASSWORD);
		}

		private String createVariableKey(String suffix) {
			return "%s_MAVEN_%s".formatted(name, suffix).toUpperCase().replaceAll("[^A-Z_]", "_");
		}
	}

	private record Credentials(
			String username,
			String password) {
	}
}
