package io.bitsensor.plugins.java.sql;

import org.apache.tomcat.jdbc.pool.DataSourceProxy;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * JdbcAspect is an aspect for automatic Jdbc interceptor register without the need to modify the java source code
 * manually.
 * <p>
 * Note that JdbcAspect only supports official MySQL and PostgreSQL drivers.
 *
 * @see MysqlJdbcInterceptor
 * @see PostgresJdbcInterceptor
 * @see org.apache.tomcat.jdbc.pool.JdbcInterceptor
 */
@Aspect
public class JdbcAspect {
    private final static Logger LOGGER = LoggerFactory.getLogger(JdbcAspect.class);

    private static final String MYSQL_JDBC_INTERCEPTOR = "io.bitsensor.plugins.java.sql.MysqlJdbcInterceptor";
    private static final String POSTGRES_JDBC_INTERCEPTOR = "io.bitsensor.plugins.java.sql.PostgresJdbcInterceptor";

    @Pointcut("within(org.apache.tomcat.jdbc.pool.DataSourceProxy) && execution(* getConnection())")
    public void getConnectionMethod() {
    }

    @Around("getConnectionMethod()")
    public Object getConnectionAround(ProceedingJoinPoint joinPoint) throws Throwable {
        DataSourceProxy dsp = (DataSourceProxy) joinPoint.getThis();

        String driverClassName = dsp.getDriverClassName();
        String jdbcInterceptors = dsp.getJdbcInterceptors();
        switch (driverClassName) {
            case "com.mysql.jdbc.Driver":
                if (jdbcInterceptors == null || !jdbcInterceptors.contains(MYSQL_JDBC_INTERCEPTOR)) {
                    jdbcInterceptors = jdbcInterceptors == null
                            ? MYSQL_JDBC_INTERCEPTOR
                            : jdbcInterceptors.concat("," + MYSQL_JDBC_INTERCEPTOR);
                    dsp.getPoolProperties().setJdbcInterceptors(jdbcInterceptors);
                }
                break;
            case "org.postgresql.Driver":
                if (jdbcInterceptors == null || !jdbcInterceptors.contains(POSTGRES_JDBC_INTERCEPTOR)) {
                    jdbcInterceptors = jdbcInterceptors == null
                            ? POSTGRES_JDBC_INTERCEPTOR
                            : jdbcInterceptors.concat("," + POSTGRES_JDBC_INTERCEPTOR);
                    dsp.getPoolProperties().setJdbcInterceptors(jdbcInterceptors);
                }
                break;
            default:
                LOGGER.error("BitSensor can not handle unknown JDBC driver class name [{}]", driverClassName);
        }

        return joinPoint.proceed();
    }
}
