package io.github.andreyzebin.gitSql.git;

import io.github.zebin.javabash.process.TextTerminal;
import io.github.zebin.javabash.sandbox.BashUtils;

import java.io.StringWriter;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

public class GitBindings {

    public static final String ALL_BRANCHES = "";
    public static final String NO_MASTER_COMMITS = notBranch("master");

    public static Optional<String> getBranch(TextTerminal bash) {
        StringWriter out = new StringWriter();
        StringWriter err = new StringWriter();
        int exec = bash.exec("git rev-parse --abbrev-ref HEAD", out::append, err::append);

        if (exec != 0) {
            throw new RuntimeException("Could not get branch: " + err);
        }

        return out.toString().lines().findFirst();
    }

    public static String periodHash(String from, String to) {
        return String.format(" %s..%s ", from, to);
    }

    public static String sinceHash(String from) {
        return periodHash(from, "");
    }

    public static String notBranch(String branch) {
        return String.format("--not %s", branch);
    }

    public static Stream<Change> filesChangedQuery(
            String branch,
            String query,
            TextTerminal bash
    ) {
        StringWriter out = new StringWriter();
        StringWriter err = new StringWriter();
        int exec = bash.exec(
                String.format("git log %s --name-status %s | grep -E '^[A-Z]\\b' | sort -k 2,2 -u", branch, query),
                out::append, err::append);

        if (exec != 0) {
            throw new RuntimeException("Could not list changes: " + err);
        }

        return out.toString().lines()
                .map(cBr -> cBr.replaceAll("\\s", ";"))
                .map(cBr -> cBr.replace(";;", ";"))
                .map(cBr -> cBr.replace(";;", ";"))
                .map(
                        cLine -> Change.builder()
                                .type(cLine.split(";")[0])
                                .file(cLine.split(";")[1])
                                .build()
                );
    }

    public static void merge(
            String headHash,
            TextTerminal bash
    ) {
        bash.eval(String.format("git merge %s", headHash));
    }

    public static void resetHard(TextTerminal bash) {
        bash.eval(String.format("git reset --hard"));
    }

    // local
    //

    public static Stream<BranchHead> getBranches(
            TextTerminal bash
    ) {
        StringWriter out = new StringWriter();
        bash.exec("git for-each-ref --format='%(refname:short)' refs/heads/", out::append,
                k -> {
                });

        return out.toString().lines()
                .map(cBr -> cBr.replaceAll("\\s", ";"))
                .map(cBr -> cBr.replace(";;", ";"))
                .map(cBr -> cBr.replace(";;", ";"))
                .map(cBr -> BranchHead.builder().hash(cBr.split(";")[0]).build());
    }

    public static Stream<BranchHead> getRemoteBranches(
            String origin,
            TextTerminal bash,
            boolean authHeader
    ) {
        StringWriter out = new StringWriter();
        bash.exec(String.format("git %s ls-remote --heads %s ",
                        (authHeader ? "-c http.extraHeader=\"Authorization: Bearer $API_TOKEN\"" : ""), origin), out::append,
                k -> {
                });

        return out.toString().lines()
                .map(cBr -> cBr.replaceAll("\\s", ";"))
                .map(cBr -> cBr.replace(";;", ";"))
                .map(cBr -> cBr.replace(";;", ";"))
                //.flatMap(cBr -> Stream.of(cBr.split(";")))
                .filter(cBr -> cBr.contains("refs/heads/"))
                .map(cBr -> cBr
                        .replace("refs/heads/", ""))
                .map(cBr -> BranchHead.builder()
                        .hash(cBr.split(";")[0])
                        .name(cBr.split(";")[1])
                        .build());
        //.filter(cBr -> !cBr.equals("HEAD"));
    }

    public static void clone(
            String target,
            TextTerminal bash,
            String uri,
            boolean isQuiet,
            boolean authHeader,
            String branch
    ) {
        bash.eval(
                String.format(
                        "git clone -c core.protectNTFS=false %s %s %s %s %s",
                        (branch != null ? "--single-branch --branch " + branch : ""),
                        (authHeader ? "-c http.extraHeader=\"Authorization: Bearer $API_TOKEN\"" : ""),
                        uri,
                        (isQuiet ? "--quiet" : ""),
                        target
                )
        );
    }

    public static void add(TextTerminal bash, Path relative) {
        bash.eval("git add " + BashUtils.escape(BashUtils.toPosix(relative)));
    }

    public static void commit(TextTerminal bash) {
        bash.eval("git commit -a -m \"msg\"");
    }

    public static Stream<Change> listStatus(TextTerminal bash) {
        StringWriter out = new StringWriter();
        StringWriter err = new StringWriter();
        int exec = bash.exec(
                "git status --porcelain --untracked-files=no",
                out::append, err::append);

        if (exec != 0) {
            throw new RuntimeException("Could not list changes: " + err);
        }

        return out.toString().lines()
                .map(cBr -> cBr.replaceAll("\\s", ";"))
                .map(cBr -> cBr.replace(";;", ";"))
                .map(cBr -> cBr.replace(";;", ";"))
                .map(
                        cLine -> Change.builder()
                                .type(cLine.split(";")[0])
                                .file(cLine.split(";")[1])
                                .build()
                );
    }

    public static boolean hasStatus(TextTerminal bash) {
        return listStatus(bash).findAny().isPresent();
    }

    public static void push(TextTerminal bash) {
        bash.eval("git push ");
    }

    public static void checkout(
            String hash,
            TextTerminal bash
    ) {
        bash.eval("git checkout " + hash);
    }

    public static void pull(
            TextTerminal bash
    ) {
        bash.eval("git pull");
    }

    public static void pullAll(
            TextTerminal bash
    ) {
        bash.eval("git pull --all");
    }

    public static Stream<Commit> commitsList(TextTerminal bash) {
        StringWriter sw = new StringWriter();
        StringWriter err = new StringWriter();
        int exec = bash.exec("git --no-pager log --pretty=%cI,%H,%at,%ce,%cn,%ae,%an,%P", sw::append, err::append);

        if (exec != 0) {
            throw new RuntimeException("Could not list commits: " + err);
        }

        return sw.toString().lines().map(cLine -> cLine.split(",")).map(split -> Commit.builder()
                .timestamp(split[0])
                .hash(split[1])
                .unixTimestamp(split[2])

                .committerEmail(split[3])
                .committerName(split[4])
                .authorEmail(split[5])
                .authorName(split[6])
                .parents(split.length > 7 ? Arrays.stream(split[7].split(" ")).toList() : List.of())
                .build());
    }

    public static void sshAgent(String pathOfKey, TextTerminal bash) {
        bash.eval("ssh-add " + (pathOfKey != null ? pathOfKey : ""));
    }

    public static Optional<String> getOrigin(TextTerminal bash) {
        StringWriter sw = new StringWriter();
        StringWriter err = new StringWriter();
        int exec = bash.exec("git config --get remote.origin.url", sw::append, err::append);

        if (exec != 0) {
            throw new RuntimeException("Could not get origin: " + err);
        }

        return sw.toString().lines().findFirst();
    }

}
