package io.github.andreyzebin.gitSql.config;

import io.github.andreyzebin.gitSql.git.GitAPI;
import io.github.zebin.javabash.sandbox.PosixPath;
import lombok.extern.slf4j.Slf4j;

import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.AbstractMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
public class ConfigVersions implements ConfigHistory, AutoCloseable {

    public static final String IO_GITHUB_VEZUVIO = "io.github.vezuvio";
    public static final long DEFAULT_LOCK_TIMEOUT_SEC = 60 * 60; // 1 hour
    private final GitAPI vc;
    private final ConfigTree ct;

    private RequestTree rt;

    public ConfigVersions(GitAPI vc, ConfigTree ct) {
        this(vc, ct, null);
    }

    public ConfigVersions(GitAPI vc, ConfigTree ct, RequestTree rt) {
        this.vc = vc;
        this.ct = ct;
        this.rt = rt;
    }

    public ConfigVersions setRt(RequestTree rt) {
        this.rt = rt;

        return this;
    }

    public void update() {
        vc.fetch();
        vc.pull();
    }

    public void commit() {
        vc.commit();
    }

    public void push() {
        vc.push();
    }

    public GitAPI getGitAPI() {
        return vc;
    }

    public String getOffset(String baseBranch) {
        return rt.getOffset(vc.getBranch().get(), baseBranch);
    }

    public void rebase(String fromBaseBranch, String toBaseBranch) {
        rt.rebase(fromBaseBranch, vc.getBranch().get(), toBaseBranch);
    }

    public void merge(String destbranch) {
        rt.merge(destbranch, vc.getBranch().get(), topVersion().get().getVersionHash());
    }

    @Override
    public String getEffectiveProperty(String hash, PosixPath leaf, String key) {
        return getHistorical(hash, () -> ct.getEffectiveProperty(leaf, key));
    }

    @Override
    public Set<Map.Entry<PosixPath, String>> getExplodedPropertyKeys(String hash) {
        return ct.getLeafs()
                .flatMap(this::flattenEffectiveProperties)
                .collect(Collectors.toSet());
    }

    private Stream<Map.Entry<PosixPath, String>> flattenEffectiveProperties(PosixPath leaf) {
        return getEffectivePropertyKeys(leaf)
                .stream()
                .map((k) -> new AbstractMap.SimpleEntry<>(leaf, k));
    }

    public Optional<String> setProperty(PosixPath leaf, String key, String value) {
        return ct.setProperty(leaf, key, value);
    }

    public Optional<String> deleteProperty(PosixPath leaf, String key) {
        return ct.deleteProperty(leaf, key);
    }

    public String getEffectiveProperty(PosixPath leaf, String key) {
        return ct.getEffectiveProperty(leaf, key);
    }

    public Set<String> getEffectivePropertyKeys(PosixPath leaf) {
        return ct.getEffectiveProperties(leaf).keySet();
    }

    private <T> T getHistorical(String hash, Supplier<T> exec) {
        String branch = vc.getBranch().get();

        if (!hash.equals(vc.listCommits().findFirst().get().getHash())) {
            try {
                vc.seek(hash);
                return exec.get();
            } finally {
                vc.reset();
                vc.setBranch(branch);
            }
        }

        return exec.get();
    }

    public Map<String, String> getEffectiveState(String hash, PosixPath leaf) {
        return getHistorical(
                hash, () -> ct.getDirectory().setupDir(
                        () -> getEffectivePropertyKeys(hash, leaf)
                                .stream()
                                .collect(Collectors.toMap(k -> k, k -> getEffectiveProperty(hash, leaf, k)))
                )
        );
    }

    public Map<String, String> getProperties(String hash, PosixPath leaf) {
        return getHistorical(hash, () -> ct.getDirectory().setupDir(() -> ct.getProperties(leaf)));
    }

    @Override
    public Map<Map.Entry<PosixPath, String>, String> getProperties(String hash) {
        return getHistorical(hash, () -> ct.getDirectory().setupDir(() -> ct.getLeafs()
                .flatMap(lf -> ct.getProperties(lf)
                        .entrySet()
                        .stream()
                        .map(pr -> new AbstractMap.SimpleEntry<>(new AbstractMap.SimpleEntry<>(lf, pr.getKey()), pr.getValue()))
                ).collect(Collectors.toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue))));
    }

    public Set<PosixPath> getExplodedLeafChanges(String from, String to) {
        log.debug("Getting exploded leaf changes...");
        Set<PosixPath> changesExploded = getExplodedChanges(from, to)
                .keySet()
                .stream()
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());
        log.debug("Getting exploded leaf changes...Finished!");
        return changesExploded;
    }

    public Map<Map.Entry<PosixPath, String>, String> getExplodedState(String hash) {
        return ct.getDirectory().setupDir(() -> {
            log.debug("Getting exploded state...");
            Map<Map.Entry<PosixPath, String>, String> historical = getHistorical(hash,
                    () -> {
                        log.debug("Getting exploded property keys...");
                        Set<Map.Entry<PosixPath, String>> propertyKeys = getExplodedPropertyKeys(hash);
                        log.debug("Getting exploded property keys...Finished!");

                        log.debug("Getting effective state...");
                        Map<Map.Entry<PosixPath, String>, String> effState = propertyKeys
                                .stream()
                                .map(k -> {
                                    String s = getEffectiveState(hash, k.getKey()).get(k.getValue());
                                    return new AbstractMap.SimpleEntry<>(k, s);
                                })
                                .filter(e -> e.getValue() != null)
                                .collect(Collectors.toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue));
                        log.debug("Getting effective state...Finished!");

                        return effState;
                    });
            log.debug("Getting exploded state...Finished!");
            return historical;
        });
    }

    @Override
    public Set<String> getEffectivePropertyKeys(String hash, PosixPath leaf) {
        return getHistorical(hash, () -> ct.getEffectiveProperties(leaf).keySet());
    }

    @Override
    public Map<String, String> getEffectiveProperties(String hash, PosixPath leaf) {
        return getHistorical(hash, () -> ct.getEffectiveProperties(leaf));
    }

    public Stream<PosixPath> getLeafs() {
        return ct.getLeafs();
    }

    @Override
    public Stream<PropertiesVersion> listVersions() {
        return vc.listCommits()
                .map(c -> PropertiesVersion.builder()
                        .versionHash(c.getHash())
                        .createdEpochSec(toEpochSeconds(c.getTimestamp()))
                        .build());
    }

    public static long toEpochSeconds(String timestamp) {
        return Instant.from(DateTimeFormatter.ISO_DATE_TIME.parse(timestamp)).toEpochMilli() / 1000;
    }

    public static long getNowEpochSec() {
        return Instant.now().getEpochSecond();
    }

    @Override
    public void close() throws Exception {

    }
}
