package io.github.andreyzebin.gitSql.config;

import io.github.andreyzebin.gitSql.git.BranchHead;
import io.github.andreyzebin.gitSql.git.GitAPI;
import io.github.andreyzebin.gitSql.git.GitFs;
import io.github.zebin.javabash.sandbox.AllFileManager;
import io.github.zebin.javabash.sandbox.PosixPath;
import lombok.extern.slf4j.Slf4j;

import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
public class RequestTree {
    private final Function<String, GitFs> originFactory;
    private final Function<GitFs, ConfigVersions> versionsFactory;
    private final String trunk;
    private final AllFileManager tt;

    public RequestTree(Function<String, GitFs> originFactory,
                       Function<GitFs, ConfigVersions> versionsFactory,
                       String trunk,
                       AllFileManager tt
    ) {
        this.originFactory = originFactory;
        this.versionsFactory = versionsFactory;
        this.trunk = trunk;
        this.tt = tt;
    }

    public Set<String> listBranches() {
        return getGitApi(trunk).listBranches().map(BranchHead::getName).collect(Collectors.toSet());
    }

    public ConfigVersions getBranch(String branch) {
        return versionsFactory.apply(getGitApi(branch));
    }

    private GitFs getGitApi(String branch) {
        return originFactory.apply(branch);
    }

    public ConfigVersions getTrunk() {
        return getBranch(trunk);
    }

    public String getOffset(String branch) {
        return getOffset(branch, trunk);
    }

    public String getOffset(String branch, String baseBranch) {
        return getBranch(branch)
                .listVersions()
                .map(ConfigHistory.PropertiesVersion::getVersionHash)
                .filter(v -> {
                    GitAPI trunkControl = getGitApi(baseBranch);
                    return trunkControl.contains(v);
                })
                .findFirst()
                .get();
    }

    public ConfigVersions createBranch(String baseBranch, String newBranch) {
        // has to be baseBranch
        GitFs gfs = getGitApi(baseBranch);
        gfs.copy(baseBranch, newBranch);
        gfs.setUpstream(newBranch, "origin");

        // TODO send event of new branch??
        return getBranch(newBranch);
    }

    /**
     * Rebase fromBranch to destinationBranch
     */
    public void rebase(String fromBranch, String branch, String destinationBranch) {
        GitFs trunkControl = getGitApi(branch);
        PosixPath location = trunkControl.getLocation();

        try {
            // TODO maybe process an event of createBranch here?
            getBranch(branch).update();

            trunkControl.setBranch(fromBranch);
            trunkControl.setBranch(destinationBranch);
            trunkControl.setBranch(branch);
            trunkControl.rebase(destinationBranch, fromBranch, branch);
            trunkControl.push(true);
        } catch (RuntimeException e) {
            log.error("Rebase failed: ", e);
        }

        StringBuilder sb = new StringBuilder();
        PosixPath current = tt.getCurrent();
        tt.go(location);
        tt.getTerminal().exec(inject("git diff --check"), sb::append, sb::append);
        tt.go(current);
        // git diff --find-renames 11b072f3f4dd433ebf5fd931f81df4cf643bb26b a98be9580c3ed2f9a0280aaf744f0aa9d0d28473
        if (sb.toString().lines().findAny().isPresent()) {
            // wait till unlocked - then rebase and try again
            trunkControl.reset();
            trunkControl.setBranch(branch);
        }
    }

    /**
     * Merge to dstBranch
     */
    public boolean merge(String dstBranch, String sourceBranch, String sourceHash) {
        GitFs trunkControl = getGitApi(dstBranch);
        PosixPath location = trunkControl.getLocation();

        tt.go(location);
        try {
            getBranch(dstBranch).update();

            trunkControl.merge(sourceHash);
            trunkControl.push();
        } catch (RuntimeException e) {
            log.error("Merge failed: ", e);
        }

        StringBuilder sb = new StringBuilder();
        tt.getTerminal().exec(inject("git diff --check"), sb::append, sb::append);
        // git diff --find-renames 11b072f3f4dd433ebf5fd931f81df4cf643bb26b a98be9580c3ed2f9a0280aaf744f0aa9d0d28473
        if (sb.toString().lines().findAny().isPresent()) {
            // wait till unlocked - then rebase and try again
            trunkControl.reset();
            trunkControl.setBranch(dstBranch);
            return false;
        }

        return true;
    }

    /**
     * Merge to Trunk
     */
    public boolean merge(String sourceBranch, String sourceHash) {
        GitFs trunkControl = getGitApi(trunk);
        PosixPath location = trunkControl.getLocation();

        tt.go(location);
        try {
            trunkControl.merge(sourceHash);
            trunkControl.push();
        } catch (RuntimeException e) {
            log.error("Merge failed: ", e);
        }

        StringBuilder sb = new StringBuilder();
        tt.getTerminal().exec(inject("git diff --check"), sb::append, sb::append);
        // git diff --find-renames 11b072f3f4dd433ebf5fd931f81df4cf643bb26b a98be9580c3ed2f9a0280aaf744f0aa9d0d28473
        if (sb.toString().lines().findAny().isPresent()) {
            // wait till unlocked - then rebase and try again
            trunkControl.reset();
            trunkControl.setBranch(trunk);
            return false;
        }

        return true;
    }

    public static String inject(String gitMerge, Object... ff) {
        return String.format(gitMerge, ff);
    }
}
