/*
 * Decompiled with CFR 0.152.
 */
package org.biojava.nbio.structure.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import org.biojava.nbio.core.util.FileDownloadUtils;
import org.biojava.nbio.core.util.InputStreamProvider;
import org.biojava.nbio.structure.PDBStatus;
import org.biojava.nbio.structure.PdbId;
import org.biojava.nbio.structure.Structure;
import org.biojava.nbio.structure.align.util.UserConfiguration;
import org.biojava.nbio.structure.io.FileParsingParameters;
import org.biojava.nbio.structure.io.StructureIOFile;
import org.rcsb.mmtf.utils.CodecUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class LocalPDBDirectory
implements StructureIOFile {
    private static final Logger logger = LoggerFactory.getLogger(LocalPDBDirectory.class);
    public static final String DEFAULT_PDB_FILE_SERVER = "https://files.wwpdb.org";
    public static final String PDB_FILE_SERVER_PROPERTY = "PDB.FILE.SERVER";
    public static final String DEFAULT_BCIF_FILE_SERVER = "https://models.rcsb.org/";
    public static final long LAST_REMEDIATION_DATE;
    private static final String LAST_REMEDIATION_DATE_STRING = "2011/07/12";
    protected static final String lineSplit;
    public static final long MIN_PDB_FILE_SIZE = 40L;
    private File path;
    private List<String> extensions = new ArrayList<String>();
    private String serverName;
    private FileParsingParameters params = new FileParsingParameters();
    private ObsoleteBehavior obsoleteBehavior;
    private FetchBehavior fetchBehavior;
    private String splitDirURL;
    private String obsoleteDirURL;
    private File splitDirPath;
    private File obsoleteDirPath;

    public LocalPDBDirectory(String path) {
        if (path == null) {
            UserConfiguration config = new UserConfiguration();
            path = config.getPdbFilePath();
            logger.debug("Initialising from system property/environment variable to path: {}", (Object)path);
        } else {
            path = FileDownloadUtils.expandUserHome((String)path);
            logger.debug("Initialising with path {}", (Object)path);
        }
        this.path = new File(path);
        this.serverName = LocalPDBDirectory.getServerName();
        this.initPaths();
        this.fetchBehavior = FetchBehavior.DEFAULT;
        this.obsoleteBehavior = ObsoleteBehavior.DEFAULT;
    }

    public LocalPDBDirectory() {
        this(null);
    }

    public void setPath(String p) {
        this.path = new File(FileDownloadUtils.expandUserHome((String)p));
        this.initPaths();
    }

    public String getPath() {
        return this.path.toString();
    }

    @Override
    public void addExtension(String s) {
        this.extensions.add(s);
    }

    @Override
    public List<String> getExtensions() {
        return Collections.unmodifiableList(this.extensions);
    }

    public void clearExtensions() {
        this.extensions.clear();
    }

    @Override
    public void setFileParsingParameters(FileParsingParameters params) {
        this.params = params;
    }

    @Override
    public FileParsingParameters getFileParsingParameters() {
        return this.params;
    }

    public void setObsoleteBehavior(ObsoleteBehavior behavior) {
        this.obsoleteBehavior = behavior;
    }

    public ObsoleteBehavior getObsoleteBehavior() {
        return this.obsoleteBehavior;
    }

    public FetchBehavior getFetchBehavior() {
        return this.fetchBehavior;
    }

    public void setFetchBehavior(FetchBehavior fetchBehavior) {
        this.fetchBehavior = fetchBehavior;
    }

    @Override
    public Structure getStructure(String filename) throws IOException {
        filename = FileDownloadUtils.expandUserHome((String)filename);
        File f = new File(filename);
        return this.getStructure(f);
    }

    public Structure getStructure(URL u) throws IOException {
        InputStreamProvider isp = new InputStreamProvider();
        InputStream inStream = isp.getInputStream(u);
        return this.getStructure(inStream);
    }

    @Override
    public Structure getStructure(File filename) throws IOException {
        InputStreamProvider isp = new InputStreamProvider();
        InputStream inStream = isp.getInputStream(filename);
        return this.getStructure(inStream);
    }

    @Override
    public Structure getStructureById(String pdbId) throws IOException {
        return this.getStructureById(new PdbId(pdbId));
    }

    @Override
    public Structure getStructureById(PdbId pdbId) throws IOException {
        InputStream inStream = this.getInputStream(pdbId);
        return this.getStructure(inStream);
    }

    public abstract Structure getStructure(InputStream var1) throws IOException;

    protected InputStream getInputStream(PdbId pdbId) throws IOException {
        File file = this.downloadStructure(pdbId);
        if (!file.exists()) {
            throw new IOException("Structure " + pdbId + " not found and unable to download.");
        }
        InputStreamProvider isp = new InputStreamProvider();
        InputStream inputStream = isp.getInputStream(file);
        return inputStream;
    }

    public void prefetchStructure(String pdbId) throws IOException {
        File file = this.downloadStructure(new PdbId(pdbId));
        if (!file.exists()) {
            throw new IOException("Structure " + pdbId + " not found and unable to download.");
        }
    }

    public boolean deleteStructure(String pdbId) throws IOException {
        return this.deleteStructure(new PdbId(pdbId));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean deleteStructure(PdbId pdbId) throws IOException {
        boolean deleted = false;
        ObsoleteBehavior obsolete = this.getObsoleteBehavior();
        this.setObsoleteBehavior(ObsoleteBehavior.FETCH_OBSOLETE);
        try {
            File existing = this.getLocalFile(pdbId);
            while (existing != null) {
                assert (existing.exists());
                if (this.getFetchBehavior() == FetchBehavior.LOCAL_ONLY) {
                    throw new RuntimeException("Refusing to delete from LOCAL_ONLY directory");
                }
                boolean success = existing.delete();
                if (success) {
                    logger.debug("Deleting " + existing.getAbsolutePath());
                }
                deleted = deleted || success;
                File parent = existing.getParentFile();
                if (parent != null && (success = parent.delete())) {
                    logger.debug("Deleting " + parent.getAbsolutePath());
                }
                existing = this.getLocalFile(pdbId);
            }
            boolean bl = deleted;
            return bl;
        }
        finally {
            this.setObsoleteBehavior(obsolete);
        }
    }

    protected File downloadStructure(PdbId pdbId) throws IOException {
        File existing = this.getLocalFile(pdbId);
        switch (this.fetchBehavior) {
            case LOCAL_ONLY: {
                if (existing == null) {
                    throw new IOException(String.format("Structure %s not found in %s and configured not to download.", pdbId, this.getPath()));
                }
                return existing;
            }
            case FETCH_FILES: {
                if (existing == null) break;
                return existing;
            }
            case FETCH_IF_OUTDATED: {
                break;
            }
            case FETCH_REMEDIATED: {
                if (existing != null) {
                    long lastModified = existing.lastModified();
                    if (lastModified < LAST_REMEDIATION_DATE) {
                        logger.warn("Replacing file {} with latest remediated (remediation of {}) file from PDB.", (Object)existing, (Object)LAST_REMEDIATION_DATE_STRING);
                        existing = null;
                        break;
                    }
                    return existing;
                }
            }
            case FORCE_DOWNLOAD: {
                existing = null;
            }
        }
        if (this.obsoleteBehavior == ObsoleteBehavior.FETCH_CURRENT) {
            String current = PDBStatus.getCurrent(pdbId.getId());
            PdbId pdbIdToDownload = null;
            pdbIdToDownload = current == null ? pdbId : new PdbId(current);
            return this.downloadStructure(pdbIdToDownload, this.splitDirURL, false, existing);
        }
        if (this.obsoleteBehavior == ObsoleteBehavior.FETCH_OBSOLETE && PDBStatus.getStatus(pdbId.getId()) == PDBStatus.Status.REMOVED) {
            return this.downloadStructure(pdbId, this.obsoleteDirURL, true, existing);
        }
        return this.downloadStructure(pdbId, this.splitDirURL, false, existing);
    }

    private File downloadStructure(PdbId pdbId, String pathOnServer, boolean obsolete, File existingFile) throws IOException {
        String id = pdbId.getId().toLowerCase();
        File dir = this.getDir(id, obsolete);
        File realFile = new File(dir, this.getFilename(id));
        String filename = this.getFilename(id);
        String ftp = filename.endsWith(".mmtf.gz") ? CodecUtils.getMmtfEntryUrl((String)id, (boolean)true, (boolean)false) : (filename.endsWith(".bcif") || filename.endsWith(".bcif.gz") ? DEFAULT_BCIF_FILE_SERVER + filename : String.format("%s%s/%s/%s", this.serverName, pathOnServer, id.substring(id.length() - 3, id.length() - 1), this.getFilename(id)));
        URL url = new URL(ftp);
        Date serverFileDate = null;
        if (existingFile != null) {
            serverFileDate = this.getLastModifiedTime(url);
            if (serverFileDate != null) {
                if (existingFile.lastModified() >= serverFileDate.getTime()) {
                    return existingFile;
                }
                logger.warn("File {} is outdated, will download new one from PDB (updated on {})", (Object)existingFile, (Object)serverFileDate.toString());
            } else {
                logger.warn("Could not determine if file {} is outdated (could not get timestamp from server). Will force redownload", (Object)existingFile);
            }
        }
        logger.info("Fetching " + ftp);
        logger.info("Writing to " + realFile);
        FileDownloadUtils.downloadFile((URL)url, (File)realFile);
        return realFile;
    }

    private Date getLastModifiedTime(URL url) {
        Date date = null;
        try {
            String lastModified = url.openConnection().getHeaderField("Last-Modified");
            logger.debug("Last modified date of server file ({}) is {}", (Object)url.toString(), (Object)lastModified);
            if (lastModified != null) {
                try {
                    date = new SimpleDateFormat("E, d MMM yyyy HH:mm:ss Z", Locale.ENGLISH).parse(lastModified);
                }
                catch (ParseException e) {
                    logger.warn("Could not parse last modified time from string '{}', no last modified time available for file {}", (Object)lastModified, (Object)url.toString());
                }
            }
        }
        catch (IOException e) {
            logger.warn("Problems while retrieving last modified time for file {}", (Object)url.toString());
        }
        return date;
    }

    protected File getDir(String pdbId, boolean obsolete) {
        boolean success;
        String middle;
        File dir = null;
        int offset = pdbId.length() - 3;
        if (obsolete) {
            middle = pdbId.substring(offset, offset + 2).toLowerCase();
            dir = new File(this.obsoleteDirPath, middle);
        } else {
            middle = pdbId.substring(offset, offset + 2).toLowerCase();
            dir = new File(this.splitDirPath, middle);
        }
        if (!dir.exists() && !(success = dir.mkdirs())) {
            logger.error("Could not create mmCIF dir {}", (Object)dir.toString());
        }
        return dir;
    }

    public File getLocalFile(String pdbId) throws IOException {
        return this.getLocalFile(new PdbId(pdbId));
    }

    public File getLocalFile(PdbId pdbId) throws IOException {
        String id = pdbId.getId();
        int offset = id.length() - 3;
        LinkedList<File> searchdirs = new LinkedList<File>();
        String middle = id.substring(offset, offset + 2).toLowerCase();
        File splitdir = new File(this.splitDirPath, middle);
        searchdirs.add(splitdir);
        if (this.getObsoleteBehavior() == ObsoleteBehavior.FETCH_OBSOLETE) {
            File obsdir = new File(this.obsoleteDirPath, middle);
            searchdirs.add(obsdir);
        }
        String[] prefixes = new String[]{"", "pdb"};
        for (File searchdir : searchdirs) {
            for (String prefix : prefixes) {
                for (String ex : this.getExtensions()) {
                    File f = new File(searchdir, prefix + id.toLowerCase() + ex);
                    if (!f.exists()) continue;
                    if (f.length() < 40L) {
                        Files.delete(f.toPath());
                        return null;
                    }
                    return f;
                }
            }
        }
        return null;
    }

    protected boolean checkFileExists(String pdbId) {
        return this.checkFileExists(new PdbId(pdbId));
    }

    protected boolean checkFileExists(PdbId pdbId) {
        try {
            File path = this.getLocalFile(pdbId);
            if (path != null) {
                return true;
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return false;
    }

    public static String getServerName() {
        String name = System.getProperty(PDB_FILE_SERVER_PROPERTY);
        if (name == null || name.trim().isEmpty()) {
            name = DEFAULT_PDB_FILE_SERVER;
            logger.debug("Using default PDB file server {}", (Object)name);
        } else {
            if (!(name.startsWith("http://") || name.startsWith("ftp://") || name.startsWith("https://"))) {
                logger.warn("Server name {} read from system property {} does not have a leading protocol string. Adding http:// to it", (Object)name, (Object)PDB_FILE_SERVER_PROPERTY);
                name = "http://" + name;
            }
            logger.info("Using PDB file server {} read from system property {}", (Object)name, (Object)PDB_FILE_SERVER_PROPERTY);
        }
        return name;
    }

    protected void initPaths() {
        int i;
        String[] split = this.getSplitDirPath();
        String[] obsolete = this.getObsoleteDirPath();
        StringBuilder splitURL = new StringBuilder("/pub/pdb");
        for (int i2 = 0; i2 < split.length; ++i2) {
            splitURL.append("/");
            splitURL.append(split[i2]);
        }
        StringBuilder obsoleteURL = new StringBuilder("/pub/pdb");
        for (i = 0; i < obsolete.length; ++i) {
            obsoleteURL.append("/");
            obsoleteURL.append(obsolete[i]);
        }
        this.splitDirURL = splitURL.toString();
        this.obsoleteDirURL = obsoleteURL.toString();
        this.splitDirPath = this.path;
        for (i = 0; i < split.length; ++i) {
            this.splitDirPath = new File(this.splitDirPath, split[i]);
        }
        this.obsoleteDirPath = this.path;
        for (i = 0; i < obsolete.length; ++i) {
            this.obsoleteDirPath = new File(this.obsoleteDirPath, obsolete[i]);
        }
    }

    protected abstract String getFilename(String var1);

    protected abstract String[] getSplitDirPath();

    protected abstract String[] getObsoleteDirPath();

    static {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy/MM/dd", Locale.US);
        long t = 0L;
        try {
            Date d = formatter.parse(LAST_REMEDIATION_DATE_STRING);
            t = d.getTime();
        }
        catch (ParseException e) {
            logger.error("Unexpected error! could not parse LAST_REMEDIATION_DATE: " + e.getMessage());
        }
        LAST_REMEDIATION_DATE = t;
        lineSplit = System.getProperty("file.separator");
    }

    public static enum FetchBehavior {
        LOCAL_ONLY,
        FETCH_FILES,
        FETCH_IF_OUTDATED,
        FETCH_REMEDIATED,
        FORCE_DOWNLOAD;

        public static final FetchBehavior DEFAULT;

        static {
            DEFAULT = FETCH_REMEDIATED;
        }
    }

    public static enum ObsoleteBehavior {
        FETCH_CURRENT,
        FETCH_OBSOLETE,
        THROW_EXCEPTION;

        public static final ObsoleteBehavior DEFAULT;

        static {
            DEFAULT = THROW_EXCEPTION;
        }
    }
}

