/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.python.embedding.tools.vfs;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.graalvm.python.embedding.tools.exec.GraalPyRunner;
import org.graalvm.python.embedding.tools.exec.SubprocessLog;

public final class VFSUtils {
    public static final String VFS_ROOT = "org.graalvm.python.vfs";
    public static final String VFS_HOME = "home";
    public static final String VFS_VENV = "venv";
    public static final String VFS_FILESLIST = "fileslist.txt";
    public static final String GRAALPY_GROUP_ID = "org.graalvm.python";
    private static final String NATIVE_IMAGE_RESOURCES_CONFIG = "{\n  \"resources\": {\n    \"includes\": [\n      {\"pattern\": \"$vfs/.*\"}\n    ]\n  }\n}\n".replace("$vfs", "org.graalvm.python.vfs");
    private static final String NATIVE_IMAGE_ARGS = "Args = -H:-CopyLanguageResources";
    private static final String INCLUDE_PREFIX = "include:";
    private static final String EXCLUDE_PREFIX = "exclude:";
    private static final boolean IS_WINDOWS = System.getProperty("os.name").startsWith("Windows");
    public static final String LAUNCHER_NAME = IS_WINDOWS ? "graalpy.exe" : "graalpy.sh";
    private static final String GRAALPY_MAIN_CLASS = "com.oracle.graal.python.shell.GraalPythonMain";

    public static void writeNativeImageConfig(Path metaInfRoot, String pluginId) throws IOException {
        Path p = metaInfRoot.resolve(Path.of("native-image", GRAALPY_GROUP_ID, pluginId));
        VFSUtils.write(p.resolve("resource-config.json"), NATIVE_IMAGE_RESOURCES_CONFIG);
        VFSUtils.write(p.resolve("native-image.properties"), NATIVE_IMAGE_ARGS);
    }

    private static void write(Path config, String txt) throws IOException {
        try {
            VFSUtils.createParentDirectories(config);
            Files.writeString(config, (CharSequence)txt, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        }
        catch (IOException e) {
            throw new IOException(String.format("failed to write %s", config), e);
        }
    }

    private static void createParentDirectories(Path path) throws IOException {
        Path parent = path.getParent();
        if (parent != null) {
            Files.createDirectories(parent, new FileAttribute[0]);
        }
    }

    public static void generateVFSFilesList(Path vfs) throws IOException {
        Path filesList = vfs.resolve(VFS_FILESLIST);
        if (!Files.isDirectory(vfs, new LinkOption[0])) {
            throw new IOException(String.format("'%s' has to exist and be a directory.\n", vfs.toString()));
        }
        HashSet<String> ret = new HashSet<String>();
        String rootPath = VFSUtils.makeDirPath(vfs.toAbsolutePath());
        int rootEndIdx = rootPath.lastIndexOf(File.separator, rootPath.lastIndexOf(File.separator) - 1);
        ret.add(rootPath.substring(rootEndIdx));
        try (Stream<Path> s = Files.walk(vfs, new FileVisitOption[0]);){
            s.forEach(p -> {
                if (Files.isDirectory(p, new LinkOption[0])) {
                    String dirPath = VFSUtils.makeDirPath(p.toAbsolutePath());
                    ret.add(dirPath.substring(rootEndIdx));
                } else if (Files.isRegularFile(p, new LinkOption[0])) {
                    ret.add(p.toAbsolutePath().toString().substring(rootEndIdx));
                }
            });
        }
        Object[] a = ret.toArray(new String[ret.size()]);
        Arrays.sort(a);
        try (FileWriter wr = new FileWriter(filesList.toFile());){
            for (Object f : a) {
                if (((String)f).charAt(0) == '\\') {
                    f = ((String)f).replace("\\", "/");
                }
                wr.write((String)f);
                wr.write("\n");
            }
        }
    }

    private static String makeDirPath(Path p) {
        Object ret = p.toString();
        if (!((String)ret).endsWith(File.separator)) {
            ret = (String)ret + File.separator;
        }
        return ret;
    }

    public static void createHome(Path homeDirectory, String graalPyVersion, List<String> includes, List<String> excludes, LauncherClassPath launcherClassPath, SubprocessLog subprocessLog, Log log) throws IOException {
        VFSUtils.trim(includes);
        VFSUtils.trim(excludes);
        Path tag = homeDirectory.resolve("tagfile");
        if (Files.isReadable(tag)) {
            List<String> lines = null;
            try {
                lines = Files.readAllLines(tag);
            }
            catch (IOException e) {
                throw new IOException(String.format("failed to read tag file %s", tag), e);
            }
            if (lines.isEmpty() || !graalPyVersion.equals(lines.get(0))) {
                log.info(String.format("Stale GraalPy home, updating to %s", graalPyVersion));
                VFSUtils.delete(homeDirectory);
            }
            if (VFSUtils.pythonHomeChanged(includes, excludes, lines)) {
                log.info(String.format("Deleting GraalPy home due to changed includes or excludes", new Object[0]));
                VFSUtils.delete(homeDirectory);
            }
        }
        try {
            if (!Files.exists(homeDirectory, new LinkOption[0])) {
                log.info(String.format("Creating GraalPy %s home in %s", graalPyVersion, homeDirectory));
                VFSUtils.createParentDirectories(homeDirectory);
                VFSUtils.copyGraalPyHome(launcherClassPath.get(), homeDirectory, includes, excludes, subprocessLog);
            }
            Files.write(tag, List.of(graalPyVersion), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
            VFSUtils.write(tag, includes, INCLUDE_PREFIX);
            VFSUtils.write(tag, excludes, EXCLUDE_PREFIX);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to copy graalpy home %s", homeDirectory), e);
        }
    }

    private static boolean pythonHomeChanged(List<String> includes, List<String> excludes, List<String> lines) {
        HashSet<String> prevIncludes = new HashSet<String>();
        HashSet<String> prevExcludes = new HashSet<String>();
        for (int i = 1; i < lines.size(); ++i) {
            String l = lines.get(i);
            if (l.startsWith(INCLUDE_PREFIX)) {
                prevIncludes.add(l.substring(INCLUDE_PREFIX.length()));
                continue;
            }
            if (!l.startsWith(EXCLUDE_PREFIX)) continue;
            prevExcludes.add(l.substring(EXCLUDE_PREFIX.length()));
        }
        boolean includeDidNotChange = prevIncludes.size() == includes.size() && prevIncludes.containsAll(includes);
        boolean excludeDidNotChange = prevExcludes.size() == excludes.size() && prevExcludes.containsAll(excludes);
        return !includeDidNotChange || !excludeDidNotChange;
    }

    private static void write(Path tag, List<String> list, String prefix) throws IOException {
        if (list != null) {
            Files.write(tag, (Iterable<? extends CharSequence>)list.stream().map(l -> prefix + l).collect(Collectors.toList()), StandardOpenOption.APPEND);
        }
    }

    public static void delete(Path dir) throws IOException {
        if (!Files.exists(dir, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> s = Files.walk(dir, new FileVisitOption[0]);){
            s.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
        }
        catch (IOException e) {
            throw new IOException(String.format("failed to delete %s", dir), e);
        }
    }

    public static void copyGraalPyHome(Set<String> classpath, Path home, Collection<String> pythonHomeIncludes, Collection<String> pythonHomeExcludes, SubprocessLog log) throws IOException, InterruptedException {
        log.log(String.format("Copying std lib to '%s'\n", home));
        String stdlibHome = null;
        String coreHome = null;
        String pathsOutputPrefix = "<=outputpaths=>";
        CollectOutputLog outputLog = new CollectOutputLog();
        GraalPyRunner.run(classpath, (SubprocessLog)outputLog, "-c", "print('" + pathsOutputPrefix + "', __graalpython__.get_python_home_paths(), sep='')");
        for (String l : outputLog.output) {
            if (!l.startsWith(pathsOutputPrefix)) continue;
            String[] s = l.substring(pathsOutputPrefix.length()).split(File.pathSeparator);
            stdlibHome = s[0];
            coreHome = s[1];
        }
        assert (stdlibHome != null);
        assert (coreHome != null);
        Path target = home.resolve("lib-graalpython");
        if (!Files.exists(target, new LinkOption[0])) {
            Files.createDirectories(target, new FileAttribute[0]);
        }
        Path source = Paths.get(coreHome, new String[0]);
        Predicate<Path> filter = f -> {
            if (Files.isDirectory(f, new LinkOption[0])) {
                if (f.getFileName().toString().equals("__pycache__") || f.getFileName().toString().equals("standalone")) {
                    return true;
                }
            } else {
                if (f.getFileName().endsWith(".py") || f.getFileName().endsWith(".txt") || f.getFileName().endsWith(".c") || f.getFileName().endsWith(".md") || f.getFileName().endsWith(".patch") || f.getFileName().endsWith(".toml") || f.getFileName().endsWith("PKG-INFO")) {
                    return true;
                }
                if (!VFSUtils.isIncluded(f.toAbsolutePath().toString(), pythonHomeIncludes)) {
                    return true;
                }
            }
            return VFSUtils.isExcluded(f.toAbsolutePath().toString(), pythonHomeExcludes);
        };
        VFSUtils.copyFolder(source, source, target, filter);
        target = home.resolve("lib-python").resolve("3");
        if (!Files.exists(target, new LinkOption[0])) {
            Files.createDirectories(target, new FileAttribute[0]);
        }
        source = Paths.get(stdlibHome, new String[0]);
        filter = f -> {
            if (Files.isDirectory(f, new LinkOption[0])) {
                if (f.getFileName().toString().equals("idlelib") || f.getFileName().toString().equals("ensurepip") || f.getFileName().toString().equals("tkinter") || f.getFileName().toString().equals("turtledemo") || f.getFileName().toString().equals("__pycache__")) {
                    return true;
                }
            } else {
                if (f.getFileName().toString().equals("libpythonvm.dll")) {
                    return true;
                }
                if (!VFSUtils.isIncluded(f.toAbsolutePath().toString(), pythonHomeIncludes)) {
                    return true;
                }
            }
            return VFSUtils.isExcluded(f.toAbsolutePath().toString(), pythonHomeExcludes);
        };
        VFSUtils.copyFolder(source, source, target, filter);
    }

    private static boolean isIncluded(String filePath, Collection<String> includes) {
        if (includes == null || includes.isEmpty()) {
            return true;
        }
        return VFSUtils.pathMatches(filePath, includes);
    }

    private static boolean isExcluded(String filePath, Collection<String> excludes) {
        if (excludes == null || excludes.isEmpty()) {
            return false;
        }
        return VFSUtils.pathMatches(filePath, excludes);
    }

    private static boolean pathMatches(String filePath, Collection<String> includes) {
        String path = filePath;
        if (File.separator.equals("\\")) {
            path = path.replaceAll("\\\\", "/");
        }
        for (String i : includes) {
            Pattern pattern = Pattern.compile(i);
            Matcher matcher = pattern.matcher(path);
            if (!matcher.matches()) continue;
            return true;
        }
        return false;
    }

    private static void copyFolder(final Path sourceRoot, Path file, final Path targetRoot, final Predicate<Path> filter) throws IOException {
        Files.walkFileTree(file, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path f, BasicFileAttributes attrs) throws IOException {
                if (filter.test(f)) {
                    return FileVisitResult.CONTINUE;
                }
                if (Files.isDirectory(f, new LinkOption[0])) {
                    VFSUtils.copyFolder(sourceRoot, f, targetRoot, filter);
                } else {
                    Path relFile = sourceRoot.relativize(f);
                    Path targetPath = targetRoot.resolve(relFile.toString());
                    Path parent = targetPath.getParent();
                    if (parent == null) {
                        return FileVisitResult.CONTINUE;
                    }
                    if (!Files.exists(parent, new LinkOption[0])) {
                        Files.createDirectories(parent, new FileAttribute[0]);
                    }
                    if (Files.exists(targetPath, new LinkOption[0])) {
                        Files.delete(targetPath);
                    }
                    Files.copy(f, targetPath, new CopyOption[0]);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void createVenv(Path venvDirectory, List<String> packages, Path laucherPath, LauncherClassPath launcherClassPath, String graalPyVersion, SubprocessLog subprocessLog, Log log) throws IOException {
        VFSUtils.generateLaunchers(laucherPath, launcherClassPath, subprocessLog, log);
        if (packages != null) {
            VFSUtils.trim(packages);
        }
        Path tag = venvDirectory.resolve("contents");
        ArrayList<String> installedPackages = new ArrayList<String>();
        if (Files.isReadable(tag)) {
            List<String> lines = null;
            try {
                lines = Files.readAllLines(tag);
            }
            catch (IOException e) {
                throw new IOException(String.format("failed to read tag file %s", tag), e);
            }
            if (lines.isEmpty() || !graalPyVersion.equals(lines.get(0))) {
                log.info(String.format("Stale GraalPy venv, updating to %s", graalPyVersion));
                VFSUtils.delete(venvDirectory);
            } else {
                for (int i = 1; i < lines.size(); ++i) {
                    installedPackages.add(lines.get(i));
                }
            }
        } else {
            log.info(String.format("Creating GraalPy %s venv", graalPyVersion));
        }
        if (!Files.exists(venvDirectory, new LinkOption[0])) {
            VFSUtils.runLauncher(laucherPath.toString(), subprocessLog, "-m", VFS_VENV, venvDirectory.toString(), "--without-pip");
            VFSUtils.runVenvBin(venvDirectory, "graalpy", subprocessLog, "-I", "-m", "ensurepip");
        }
        if (packages != null) {
            VFSUtils.deleteUnwantedPackages(venvDirectory, packages, installedPackages, subprocessLog);
            VFSUtils.installWantedPackages(venvDirectory, packages, installedPackages, subprocessLog);
        }
        try {
            Files.write(tag, List.of(graalPyVersion), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
            Files.write(tag, packages, StandardOpenOption.APPEND);
        }
        catch (IOException e) {
            throw new IOException(String.format("failed to write tag file %s", tag), e);
        }
    }

    private static void generateLaunchers(Path laucherPath, LauncherClassPath launcherClassPath, SubprocessLog subprocessLog, Log log) throws IOException {
        if (!Files.exists(laucherPath, new LinkOption[0])) {
            File tmp;
            log.info("Generating GraalPy launchers");
            VFSUtils.createParentDirectories(laucherPath);
            Path java = Paths.get(System.getProperty("java.home"), "bin", "java");
            String classpath = String.join((CharSequence)File.pathSeparator, launcherClassPath.get());
            if (!IS_WINDOWS) {
                String script = String.format("#!/usr/bin/env bash\n%s -classpath %s %s --python.Executable=\"$0\" \"$@\"\n", java, String.join((CharSequence)File.pathSeparator, classpath), GRAALPY_MAIN_CLASS);
                try {
                    Files.writeString(laucherPath, (CharSequence)script, new OpenOption[0]);
                    Set<PosixFilePermission> perms = Files.getPosixFilePermissions(laucherPath, new LinkOption[0]);
                    perms.addAll(List.of(PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_EXECUTE));
                    Files.setPosixFilePermissions(laucherPath, perms);
                }
                catch (IOException e) {
                    throw new IOException(String.format("failed to create launcher %s", laucherPath), e);
                }
            }
            String script = String.format("import os, shutil, struct, venv\nfrom pathlib import Path\nvl = os.path.join(venv.__path__[0], 'scripts', 'nt', 'graalpy.exe')\ntl = os.path.join(r'%s')\nos.makedirs(Path(tl).parent.absolute(), exist_ok=True)\nshutil.copy(vl, tl)\ncmd = r'%s -classpath \"%s\" %s'\npyvenvcfg = os.path.join(os.path.dirname(tl), \"pyvenv.cfg\")\nwith open(pyvenvcfg, 'w', encoding='utf-8') as f:\n    f.write('venvlauncher_command = ')\n    f.write(cmd)\n", laucherPath, java, classpath, GRAALPY_MAIN_CLASS);
            try {
                tmp = File.createTempFile("create_launcher", ".py");
            }
            catch (IOException e) {
                throw new IOException("failed to create tmp launcher", e);
            }
            tmp.deleteOnExit();
            try (FileWriter wr = new FileWriter(tmp);){
                wr.write(script);
            }
            catch (IOException e) {
                throw new IOException(String.format("failed to write tmp launcher %s", tmp), e);
            }
            try {
                GraalPyRunner.run(classpath, subprocessLog, tmp.getAbsolutePath());
            }
            catch (InterruptedException e) {
                throw new IOException(String.format("failed to run Graalpy launcher", new Object[0]), e);
            }
        }
    }

    private static void installWantedPackages(Path venvDirectory, List<String> packages, List<String> installedPackages, SubprocessLog subprocessLog) throws IOException {
        HashSet<String> pkgsToInstall = new HashSet<String>(packages);
        pkgsToInstall.removeAll(installedPackages);
        if (pkgsToInstall.isEmpty()) {
            return;
        }
        VFSUtils.runPip(venvDirectory, "install", subprocessLog, pkgsToInstall.toArray(new String[pkgsToInstall.size()]));
    }

    private static void deleteUnwantedPackages(Path venvDirectory, List<String> packages, List<String> installedPackages, SubprocessLog subprocessLog) throws IOException {
        ArrayList<String> args = new ArrayList<String>(installedPackages);
        args.removeAll(packages);
        if (args.isEmpty()) {
            return;
        }
        args.add(0, "-y");
        VFSUtils.runPip(venvDirectory, "uninstall", subprocessLog, args.toArray(new String[args.size()]));
    }

    private static void runLauncher(String launcherPath, SubprocessLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runLauncher(launcherPath, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute launcher command %s", List.of(args)));
        }
    }

    private static void runPip(Path venvDirectory, String command, SubprocessLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runPip(venvDirectory, command, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute pip %s", List.of(args)), e);
        }
    }

    private static void runVenvBin(Path venvDirectory, String bin, SubprocessLog log, String ... args) throws IOException {
        try {
            GraalPyRunner.runVenvBin(venvDirectory, bin, log, args);
        }
        catch (IOException | InterruptedException e) {
            throw new IOException(String.format("failed to execute venv %s", List.of(args)), e);
        }
    }

    public static List<String> trim(List<String> l) {
        Iterator<String> it = l.iterator();
        while (it.hasNext()) {
            String p = it.next();
            if (p != null && !p.trim().isEmpty()) continue;
            it.remove();
        }
        return l;
    }

    public static interface Log {
        public void info(String var1);
    }

    @FunctionalInterface
    public static interface LauncherClassPath {
        public Set<String> get() throws IOException;
    }

    private static class CollectOutputLog
    implements SubprocessLog {
        private final List<String> output = new ArrayList<String>();

        private CollectOutputLog() {
        }

        @Override
        public void subProcessOut(CharSequence var1) {
            this.output.add(var1.toString());
        }

        @Override
        public void subProcessErr(CharSequence var1) {
            System.err.println(var1);
        }

        @Override
        public void log(CharSequence var1) {
        }
    }
}

