package dev.shortloop.agent;

import dev.shortloop.agent.buffer.DiscoveredApiBufferManager;
import dev.shortloop.agent.buffer.RegisteredApiBufferManager;
import dev.shortloop.agent.config.ConfigManager;
import dev.shortloop.agent.config.SimpleConfigManager;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.web.filter.OncePerRequestFilter;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.Random;


@Configuration(proxyBeanMethods = false)
@ConditionalOnProperty(name = "shortloop.enabled", havingValue = "true")
@Import(ShortloopHttpConnectionConfig.class)
public class ShortloopAutoConfiguration {
    @Value("${shortloop.url:}")
    private String url;

    @Value("${shortloop.applicationName:}")
    private String userApplicationName;

    @Value("${shortloop.loggingEnabled:false}")
    private boolean loggingEnabled;

    @Value("${shortloop.logLevel:ERROR}")
    private String logLevel;


    @Bean
    @Qualifier("shortloop-filter")
    @ConditionalOnProperty(prefix = "shortloop", name = {"url"})
    OncePerRequestFilter getShortloopSpringFilter(ShortloopHttpConnectionConfig shortloopHttpConnectionConfig) {
        SDKLogger.LogLevel logLevel = null == this.logLevel ? SDKLogger.LogLevel.ERROR : SDKLogger.LogLevel.valueOf(this.logLevel);
        SDKLogger sdkLogger = new SDKLogger(loggingEnabled, logLevel);

        if (null == url || url.length() == 0) {
            String message = "Mandatory Config `shortloop.url` is missing";
            System.out.println(message);
            sdkLogger.forceLog(message);
            return new ShortloopNoOpFilter();
        }
        if (null == userApplicationName || userApplicationName.length() == 0) {
            String message = "Mandatory Config `shortloop.applicationName` is missing";
            System.out.println(message);
            sdkLogger.forceLog(message);
            return new ShortloopNoOpFilter();
        }
        url = url.trim();
        userApplicationName = userApplicationName.trim();

        Random random = new Random();
        int randomId = random.nextInt(1000);
        String agentId = String.valueOf(randomId);

        ShortloopHTTPConnection shortloopHTTPConnection = new ShortloopHTTPConnection(url, userApplicationName, agentId, shortloopHttpConnectionConfig, sdkLogger);

        ConfigManager configManager = new SimpleConfigManager(url, userApplicationName, agentId, shortloopHTTPConnection, sdkLogger);
        configManager.init();

        RegisteredApiBufferManager registeredApiBufferManager = new RegisteredApiBufferManager(configManager, shortloopHTTPConnection, url, sdkLogger);
        registeredApiBufferManager.init();

        DiscoveredApiBufferManager discoveredApiBufferManager = new DiscoveredApiBufferManager(configManager, shortloopHTTPConnection, url, sdkLogger);
        discoveredApiBufferManager.init();

        ShortloopApiProcessor shortloopApiProcessor = new ShortloopApiProcessor(registeredApiBufferManager, discoveredApiBufferManager, sdkLogger);
        ShortloopSpringFilter shortloopSpringFilter = new ShortloopSpringFilter(configManager, shortloopApiProcessor, userApplicationName, sdkLogger);
        shortloopSpringFilter.init();
        String message = String.format("Shortloop Initialised!\napplicationName:: %s\nurl:: %s\nagentId:: %s\nSDK Version:: %s.%s",
                userApplicationName, url, agentId, SDKVersion.MAJOR_VERSION, SDKVersion.MINOR_VERSION);
        System.out.println(message);
        sdkLogger.forceLog(message);
        return shortloopSpringFilter;
    }

}
