/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.remoteconfig.internal;

import android.util.Log;
import androidx.annotation.GuardedBy;
import androidx.annotation.VisibleForTesting;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;
import com.google.firebase.remoteconfig.ConfigUpdate;
import com.google.firebase.remoteconfig.ConfigUpdateListener;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigClientException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigServerException;
import com.google.firebase.remoteconfig.internal.ConfigCacheClient;
import com.google.firebase.remoteconfig.internal.ConfigContainer;
import com.google.firebase.remoteconfig.internal.ConfigFetchHandler;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.json.JSONException;
import org.json.JSONObject;

public class ConfigAutoFetch {
    private static final int MAXIMUM_FETCH_ATTEMPTS = 3;
    private static final String TEMPLATE_VERSION_KEY = "latestTemplateVersionNumber";
    private static final String REALTIME_DISABLED_KEY = "featureDisabled";
    @GuardedBy(value="this")
    private final Set<ConfigUpdateListener> eventListeners;
    private final HttpURLConnection httpURLConnection;
    private final ConfigFetchHandler configFetchHandler;
    private final ConfigCacheClient activatedCache;
    private final ConfigUpdateListener retryCallback;
    private final ScheduledExecutorService scheduledExecutorService;
    private final Random random;

    public ConfigAutoFetch(HttpURLConnection httpURLConnection, ConfigFetchHandler configFetchHandler, ConfigCacheClient activatedCache, Set<ConfigUpdateListener> eventListeners, ConfigUpdateListener retryCallback, ScheduledExecutorService scheduledExecutorService) {
        this.httpURLConnection = httpURLConnection;
        this.configFetchHandler = configFetchHandler;
        this.activatedCache = activatedCache;
        this.eventListeners = eventListeners;
        this.retryCallback = retryCallback;
        this.scheduledExecutorService = scheduledExecutorService;
        this.random = new Random();
    }

    private synchronized void propagateErrors(FirebaseRemoteConfigException exception) {
        for (ConfigUpdateListener listener2 : this.eventListeners) {
            listener2.onError(exception);
        }
    }

    private synchronized void executeAllListenerCallbacks(ConfigUpdate configUpdate) {
        for (ConfigUpdateListener listener2 : this.eventListeners) {
            listener2.onUpdate(configUpdate);
        }
    }

    private synchronized boolean isEventListenersEmpty() {
        return this.eventListeners.isEmpty();
    }

    private String parseAndValidateConfigUpdateMessage(String message) {
        int left = message.indexOf(123);
        int right = message.lastIndexOf(125);
        if (left < 0 || right < 0) {
            return "";
        }
        return left >= right ? "" : message.substring(left, right + 1);
    }

    @VisibleForTesting
    public void listenForNotifications() {
        if (this.httpURLConnection == null) {
            return;
        }
        try {
            InputStream inputStream = this.httpURLConnection.getInputStream();
            this.handleNotifications(inputStream);
            inputStream.close();
        }
        catch (IOException ex) {
            Log.d("FirebaseRemoteConfig", "Stream was cancelled due to an exception. Retrying the connection...", ex);
        }
        finally {
            this.httpURLConnection.disconnect();
        }
    }

    private void handleNotifications(InputStream inputStream) throws IOException {
        String partialConfigUpdateMessage;
        BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, "utf-8"));
        String currentConfigUpdateMessage = "";
        while ((partialConfigUpdateMessage = reader.readLine()) != null) {
            currentConfigUpdateMessage = currentConfigUpdateMessage + partialConfigUpdateMessage;
            if (!partialConfigUpdateMessage.contains("}") || (currentConfigUpdateMessage = this.parseAndValidateConfigUpdateMessage(currentConfigUpdateMessage)).isEmpty()) continue;
            try {
                JSONObject jsonObject = new JSONObject(currentConfigUpdateMessage);
                if (jsonObject.has(REALTIME_DISABLED_KEY) && jsonObject.getBoolean(REALTIME_DISABLED_KEY)) {
                    this.retryCallback.onError(new FirebaseRemoteConfigServerException("The server is temporarily unavailable. Try again in a few minutes.", FirebaseRemoteConfigException.Code.CONFIG_UPDATE_UNAVAILABLE));
                    break;
                }
                if (this.isEventListenersEmpty()) break;
                if (jsonObject.has(TEMPLATE_VERSION_KEY)) {
                    long oldTemplateVersion = this.configFetchHandler.getTemplateVersionNumber();
                    long targetTemplateVersion = jsonObject.getLong(TEMPLATE_VERSION_KEY);
                    if (targetTemplateVersion > oldTemplateVersion) {
                        this.autoFetch(3, targetTemplateVersion);
                    }
                }
            }
            catch (JSONException ex) {
                this.propagateErrors(new FirebaseRemoteConfigClientException("Unable to parse config update message.", ex.getCause(), FirebaseRemoteConfigException.Code.CONFIG_UPDATE_MESSAGE_INVALID));
                Log.e("FirebaseRemoteConfig", "Unable to parse latest config update message.", ex);
            }
            currentConfigUpdateMessage = "";
        }
        reader.close();
        inputStream.close();
    }

    private void autoFetch(final int remainingAttempts, final long targetVersion) {
        if (remainingAttempts == 0) {
            this.propagateErrors(new FirebaseRemoteConfigServerException("Unable to fetch the latest version of the template.", FirebaseRemoteConfigException.Code.CONFIG_UPDATE_NOT_FETCHED));
            return;
        }
        int timeTillFetch = this.random.nextInt(4);
        this.scheduledExecutorService.schedule(new Runnable(){

            @Override
            public void run() {
                ConfigAutoFetch.this.fetchLatestConfig(remainingAttempts, targetVersion);
            }
        }, (long)timeTillFetch, TimeUnit.SECONDS);
    }

    @VisibleForTesting
    public synchronized Task<Void> fetchLatestConfig(int remainingAttempts, long targetVersion) {
        int remainingAttemptsAfterFetch = remainingAttempts - 1;
        int currentAttemptNumber = 3 - remainingAttemptsAfterFetch;
        Task<ConfigFetchHandler.FetchResponse> fetchTask = this.configFetchHandler.fetchNowWithTypeAndAttemptNumber(ConfigFetchHandler.FetchType.REALTIME, currentAttemptNumber);
        Task<ConfigContainer> activatedConfigsTask = this.activatedCache.get();
        return Tasks.whenAllComplete(fetchTask, activatedConfigsTask).continueWithTask(this.scheduledExecutorService, listOfUnusedCompletedTasks -> {
            Set<String> updatedKeys;
            if (!fetchTask.isSuccessful()) {
                return Tasks.forException(new FirebaseRemoteConfigClientException("Failed to auto-fetch config update.", fetchTask.getException()));
            }
            if (!activatedConfigsTask.isSuccessful()) {
                return Tasks.forException(new FirebaseRemoteConfigClientException("Failed to get activated config for auto-fetch", activatedConfigsTask.getException()));
            }
            ConfigFetchHandler.FetchResponse fetchResponse = (ConfigFetchHandler.FetchResponse)fetchTask.getResult();
            ConfigContainer activatedConfigs = (ConfigContainer)activatedConfigsTask.getResult();
            if (!ConfigAutoFetch.fetchResponseIsUpToDate(fetchResponse, targetVersion).booleanValue()) {
                Log.d("FirebaseRemoteConfig", "Fetched template version is the same as SDK's current version. Retrying fetch.");
                this.autoFetch(remainingAttemptsAfterFetch, targetVersion);
                return Tasks.forResult(null);
            }
            if (fetchResponse.getFetchedConfigs() == null) {
                Log.d("FirebaseRemoteConfig", "The fetch succeeded, but the backend had no updates.");
                return Tasks.forResult(null);
            }
            if (activatedConfigs == null) {
                activatedConfigs = ConfigContainer.newBuilder().build();
            }
            if ((updatedKeys = activatedConfigs.getChangedParams(fetchResponse.getFetchedConfigs())).isEmpty()) {
                Log.d("FirebaseRemoteConfig", "Config was fetched, but no params changed.");
                return Tasks.forResult(null);
            }
            ConfigUpdate configUpdate = ConfigUpdate.create(updatedKeys);
            this.executeAllListenerCallbacks(configUpdate);
            return Tasks.forResult(null);
        });
    }

    private static Boolean fetchResponseIsUpToDate(ConfigFetchHandler.FetchResponse response, long lastKnownVersion) {
        if (response.getFetchedConfigs() != null) {
            return response.getFetchedConfigs().getTemplateVersionNumber() >= lastKnownVersion;
        }
        return response.getStatus() == 1;
    }
}

