package fi.evolver.script;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Path;

public class Download {

	/**
	 * Downloads a file from the provided URL to a temporary file.
	 * The temporary file is scheduled for deletion on JVM exit.
	 *
	 * @return Path to the downloaded temporary file
	 * @param uri
	 */
	public static Path intoTempFile(URI uri) {
		try (Step step = Step.start("Download temp file")) {
			String filename = uri.getPath().substring(uri.getPath().lastIndexOf('/') + 1);
			int lastDotIndex = filename.lastIndexOf('.');
			String prefix = lastDotIndex > -1 ? filename.substring(0, lastDotIndex) : filename;
			String suffix = lastDotIndex > -1 ? filename.substring(lastDotIndex) : ".tmp";
			Path tempFile = FileUtils.createTempFile(prefix, suffix);
			intoFile(uri, tempFile);
			return tempFile;
		}
	}

	public static void intoFile(URI uri, Path target) {
		try (
				Step step = Step.start("Download file from %s to %s".formatted(uri, target));
				HttpClient client = HttpClient.newBuilder().followRedirects(HttpClient.Redirect.NORMAL).build();
		) {
			HttpRequest request = HttpRequest.newBuilder().uri(uri).build();
			HttpResponse<Path> response = client.send(request, HttpResponse.BodyHandlers.ofFile(target));
			if (response.statusCode() != 200) {
				throw new RuntimeException("Failed to download file: HTTP error code " + response.statusCode());
			}
		} catch (IOException | InterruptedException e) {
			throw new RuntimeException(e);
		}
	}

	public static String contentString(URI uri) {
		try (
				Step step = Step.start("Download contents from %s".formatted(uri));
				HttpClient client = HttpClient.newBuilder().followRedirects(HttpClient.Redirect.NORMAL).build();
		) {
			HttpRequest request = HttpRequest.newBuilder().uri(uri).build();
			HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
			if (response.statusCode() != 200) {
				throw new RuntimeException("Failed to download file: HTTP error code " + response.statusCode());
			}
			return response.body();
		} catch (IOException | InterruptedException e) {
			throw new RuntimeException(e);
		}
	}

	public static byte[] contentBytes(URI uri) {
		try (
				Step step = Step.start("Download bytes from %s".formatted(uri));
				HttpClient client = HttpClient.newBuilder().followRedirects(HttpClient.Redirect.NORMAL).build();
		) {
			HttpRequest request = HttpRequest.newBuilder().uri(uri).build();
			HttpResponse<byte[]> response = client.send(request, HttpResponse.BodyHandlers.ofByteArray());
			if (response.statusCode() != 200) {
				throw new RuntimeException("Failed to download file: HTTP error code " + response.statusCode());
			}
			return response.body();
		} catch (IOException | InterruptedException e) {
			throw new RuntimeException(e);
		}
	}
}
