/*
 * Decompiled with CFR 0.152.
 */
package org.evomaster.client.java.controller.internal.db;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.evomaster.client.java.controller.api.dto.database.schema.ColumnDto;
import org.evomaster.client.java.controller.api.dto.database.schema.DatabaseType;
import org.evomaster.client.java.controller.api.dto.database.schema.DbSchemaDto;
import org.evomaster.client.java.controller.api.dto.database.schema.ForeignKeyDto;
import org.evomaster.client.java.controller.api.dto.database.schema.TableCheckExpressionDto;
import org.evomaster.client.java.controller.api.dto.database.schema.TableDto;
import org.evomaster.client.java.controller.internal.db.constraint.DbTableCheckExpression;
import org.evomaster.client.java.controller.internal.db.constraint.DbTableConstraint;
import org.evomaster.client.java.controller.internal.db.constraint.DbTableUniqueConstraint;
import org.evomaster.client.java.controller.internal.db.constraint.TableConstraintExtractor;
import org.evomaster.client.java.controller.internal.db.constraint.TableConstraintExtractorFactory;
import org.evomaster.client.java.utils.SimpleLogger;

public class SchemaExtractor {
    public static boolean validate(DbSchemaDto schema) throws IllegalArgumentException {
        Objects.requireNonNull(schema);
        for (TableDto table : schema.tables) {
            for (ColumnDto column : table.columns) {
                SchemaExtractor.checkForeignKeyToAutoIncrementPresent(schema, table, column);
                SchemaExtractor.checkForeignKeyToAutoIncrementMissing(schema, table, column);
            }
        }
        return true;
    }

    private static void checkForeignKeyToAutoIncrementMissing(DbSchemaDto schema, TableDto table, ColumnDto column) {
        if (column.foreignKeyToAutoIncrement) {
            return;
        }
        Optional<ForeignKeyDto> fk = table.foreignKeys.stream().filter(it -> it.sourceColumns.contains(column.name)).findFirst();
        if (!fk.isPresent()) {
            return;
        }
        Optional<TableDto> targetTable = schema.tables.stream().filter(t -> t.name.equals(((ForeignKeyDto)fk.get()).targetTable)).findFirst();
        if (!targetTable.isPresent()) {
            throw new IllegalArgumentException("Foreign key in table " + table.name + " pointing to non-existent table " + fk.get().targetTable);
        }
        List pks = targetTable.get().columns.stream().filter(c -> c.primaryKey).collect(Collectors.toList());
        if (pks.isEmpty()) {
            throw new IllegalArgumentException("No PK in table " + targetTable.get().name + " that has FKs pointing to it");
        }
        for (ColumnDto pk : pks) {
            if (!pk.autoIncrement && !pk.foreignKeyToAutoIncrement) continue;
            throw new IllegalArgumentException("Column " + pk.name + " in table " + pk.table + " is auto-increment, although FK pointing to it does not mark it as autoincrement in " + column.name + " in " + table.name);
        }
    }

    private static void checkForeignKeyToAutoIncrementPresent(DbSchemaDto schema, TableDto table, ColumnDto column) {
        if (!column.foreignKeyToAutoIncrement) {
            return;
        }
        Optional<ForeignKeyDto> fk = table.foreignKeys.stream().filter(it -> it.sourceColumns.contains(column.name)).findFirst();
        if (!fk.isPresent()) {
            throw new IllegalArgumentException("No foreign key constraint for marked column " + column.name + " in table " + table.name);
        }
        Optional<TableDto> targetTable = schema.tables.stream().filter(t -> t.name.equals(((ForeignKeyDto)fk.get()).targetTable)).findFirst();
        if (!targetTable.isPresent()) {
            throw new IllegalArgumentException("Foreign key in table " + table.name + " pointing to non-existent table " + fk.get().targetTable);
        }
        List pks = targetTable.get().columns.stream().filter(c -> c.primaryKey).collect(Collectors.toList());
        if (pks.size() != 1) {
            throw new IllegalArgumentException("There must be only 1 PK in table " + targetTable.get().name + " pointed by the FK-to-autoincrement " + column.name + " in " + table.name + ". However, there were: " + pks.size());
        }
        ColumnDto pk = (ColumnDto)pks.get(0);
        if (!pk.autoIncrement && !pk.foreignKeyToAutoIncrement) {
            throw new IllegalArgumentException("Column " + pk.name + " in table " + pk.table + " is not auto-increment, although FK pointing to it does mark itas autoincrement in " + column.name + " in " + table.name);
        }
    }

    public static DbSchemaDto extract(Connection connection) throws Exception {
        Objects.requireNonNull(connection);
        DbSchemaDto schemaDto = new DbSchemaDto();
        try {
            schemaDto.name = connection.getSchema();
        }
        catch (AbstractMethodError | Exception e) {
            schemaDto.name = "public";
        }
        DatabaseMetaData md = connection.getMetaData();
        String protocol = md.getURL();
        DatabaseType dt = DatabaseType.OTHER;
        if (protocol.contains(":h2")) {
            dt = DatabaseType.H2;
        } else if (protocol.contains(":derby")) {
            dt = DatabaseType.DERBY;
        } else if (protocol.contains(":postgresql")) {
            dt = DatabaseType.POSTGRES;
        }
        schemaDto.databaseType = dt;
        schemaDto.name = schemaDto.name.toUpperCase();
        ResultSet tables = md.getTables(null, schemaDto.name, null, new String[]{"TABLE"});
        HashSet<String> tableNames = new HashSet<String>();
        if (!tables.next()) {
            tables.close();
            schemaDto.name = schemaDto.name.toLowerCase();
            tables = md.getTables(null, schemaDto.name, null, new String[]{"TABLE"});
            if (tables.next()) {
                do {
                    SchemaExtractor.handleTableEntry(schemaDto, md, tables, tableNames);
                } while (tables.next());
            }
        } else {
            do {
                SchemaExtractor.handleTableEntry(schemaDto, md, tables, tableNames);
            } while (tables.next());
        }
        tables.close();
        SchemaExtractor.addForeignKeyToAutoIncrement(schemaDto);
        SchemaExtractor.addConstraints(connection, dt, schemaDto);
        assert (SchemaExtractor.validate(schemaDto));
        return schemaDto;
    }

    public static void addUniqueConstraintToColumn(String tableName, TableDto tableDto, String columnName) {
        ColumnDto columnDto = tableDto.columns.stream().filter(c -> c.name.equals(columnName)).findAny().orElse(null);
        if (columnDto == null) {
            throw new IllegalArgumentException("Missing column DTO for column:" + tableName + "." + columnName);
        }
        columnDto.unique = true;
    }

    private static void addConstraints(Connection connection, DatabaseType dt, DbSchemaDto schemaDto) throws SQLException {
        TableConstraintExtractor constraintExtractor = TableConstraintExtractorFactory.buildConstraintExtractor(dt);
        if (constraintExtractor != null) {
            List<DbTableConstraint> dbTableConstraints = constraintExtractor.extract(connection, schemaDto);
            SchemaExtractor.addConstraints(schemaDto, dbTableConstraints);
        } else {
            SimpleLogger.uniqueWarn("WARNING: EvoMaster cannot extract constraints from database " + (Object)((Object)dt));
        }
    }

    private static void addConstraints(DbSchemaDto schemaDto, List<DbTableConstraint> constraintList) {
        for (DbTableConstraint constraint : constraintList) {
            String tableName = constraint.getTableName();
            TableDto tableDto = schemaDto.tables.stream().filter(t -> t.name.equalsIgnoreCase(tableName)).findFirst().orElse(null);
            if (constraint instanceof DbTableCheckExpression) {
                TableCheckExpressionDto constraintDto = new TableCheckExpressionDto();
                DbTableCheckExpression tableCheckExpression = (DbTableCheckExpression)constraint;
                constraintDto.sqlCheckExpression = tableCheckExpression.getSqlCheckExpression();
                tableDto.tableCheckExpressions.add(constraintDto);
                continue;
            }
            if (constraint instanceof DbTableUniqueConstraint) {
                DbTableUniqueConstraint tableUniqueConstraint = (DbTableUniqueConstraint)constraint;
                for (String columnName : tableUniqueConstraint.getUniqueColumnNames()) {
                    SchemaExtractor.addUniqueConstraintToColumn(tableName, tableDto, columnName);
                }
                continue;
            }
            throw new RuntimeException("Unknown constraint type " + constraint.getClass().getName());
        }
    }

    private static void handleTableEntry(DbSchemaDto schemaDto, DatabaseMetaData md, ResultSet tables, Set<String> tableNames) throws SQLException {
        TableDto tableDto = new TableDto();
        schemaDto.tables.add(tableDto);
        tableDto.name = tables.getString("TABLE_NAME");
        if (tableNames.contains(tableDto.name)) {
            throw new IllegalArgumentException("Cannot handle repeated table " + tableDto.name + " in schema");
        }
        tableNames.add(tableDto.name);
        HashSet<String> pks = new HashSet<String>();
        TreeMap<Integer, String> primaryKeySequence = new TreeMap<Integer, String>();
        ResultSet rsPK = md.getPrimaryKeys(null, null, tableDto.name);
        while (rsPK.next()) {
            String pkColumnName = rsPK.getString("COLUMN_NAME");
            short positionInPrimaryKey = rsPK.getShort("KEY_SEQ");
            pks.add(pkColumnName);
            int pkIndex = positionInPrimaryKey - 1;
            primaryKeySequence.put(pkIndex, pkColumnName);
        }
        rsPK.close();
        tableDto.primaryKeySequence.addAll(primaryKeySequence.values());
        ResultSet columns = md.getColumns(null, schemaDto.name, tableDto.name, null);
        HashSet<String> columnNames = new HashSet<String>();
        while (columns.next()) {
            ColumnDto columnDto = new ColumnDto();
            tableDto.columns.add(columnDto);
            columnDto.table = tableDto.name;
            columnDto.name = columns.getString("COLUMN_NAME");
            if (columnNames.contains(columnDto.name)) {
                throw new IllegalArgumentException("Cannot handle repeated column " + columnDto.name + " in table " + tableDto.name);
            }
            columnNames.add(columnDto.name);
            columnDto.type = columns.getString("TYPE_NAME");
            columnDto.size = columns.getInt("COLUMN_SIZE");
            columnDto.nullable = columns.getBoolean("IS_NULLABLE");
            columnDto.autoIncrement = columns.getBoolean("IS_AUTOINCREMENT");
            columnDto.primaryKey = pks.contains(columnDto.name);
        }
        columns.close();
        ResultSet fks = md.getImportedKeys(null, null, tableDto.name);
        while (fks.next()) {
            ForeignKeyDto fkDto = new ForeignKeyDto();
            fkDto.sourceColumns.add(fks.getString("FKCOLUMN_NAME"));
            fkDto.targetTable = fks.getString("PKTABLE_NAME");
            tableDto.foreignKeys.add(fkDto);
        }
        fks.close();
    }

    private static void addForeignKeyToAutoIncrement(DbSchemaDto schema) {
        for (TableDto tableDto : schema.tables) {
            for (ColumnDto columnDto : tableDto.columns) {
                if (!SchemaExtractor.isFKToAutoIncrementColumn(schema, tableDto, columnDto.name)) continue;
                columnDto.foreignKeyToAutoIncrement = true;
            }
        }
    }

    private static TableDto getTable(DbSchemaDto schema, String tableName) {
        TableDto tableDto = schema.tables.stream().filter(t -> t.name.equalsIgnoreCase(tableName)).findFirst().orElse(null);
        return tableDto;
    }

    private static ColumnDto getColumn(TableDto table, String columnName) {
        ColumnDto columnDto = table.columns.stream().filter(c -> c.name.equalsIgnoreCase(columnName)).findFirst().orElse(null);
        return columnDto;
    }

    private static boolean isFKToAutoIncrementColumn(DbSchemaDto schema, TableDto tableDto, String columnName) {
        Objects.requireNonNull(schema);
        Objects.requireNonNull(tableDto);
        Objects.requireNonNull(columnName);
        if (!tableDto.foreignKeys.stream().anyMatch(fk -> fk.sourceColumns.stream().anyMatch(s -> s.equalsIgnoreCase(columnName)))) {
            return false;
        }
        ColumnDto columnDto = SchemaExtractor.getColumn(tableDto, columnName);
        if (columnDto.autoIncrement) {
            return false;
        }
        for (ForeignKeyDto fk2 : tableDto.foreignKeys) {
            if (!fk2.sourceColumns.stream().anyMatch(s -> s.equalsIgnoreCase(columnName))) continue;
            int positionInFKSequence = fk2.sourceColumns.indexOf(columnName);
            TableDto targetTableDto = SchemaExtractor.getTable(schema, fk2.targetTable);
            String targetColumnName = targetTableDto.primaryKeySequence.get(positionInFKSequence);
            ColumnDto targetColumnDto = SchemaExtractor.getColumn(targetTableDto, targetColumnName);
            if (!targetColumnDto.autoIncrement && !SchemaExtractor.isFKToAutoIncrementColumn(schema, targetTableDto, targetColumnName)) continue;
            return true;
        }
        return false;
    }
}

